(ns facetious.authentication.twitter
  (:require
   [facetious.helpers :as helpers]
   [oauth.client :as oauth]
   [environ.core :refer [env]]
   [liberator.core :refer [defresource log!]]
   [liberator.representation :refer [ring-response as-response]]
   [ring.util.response :as util]))

;; entry point is auth, with type authorize, authenticate or force-login (Twitter sign-in flow).

(defn make-consumer [& {type :type}]
  {:pre [(contains? #{:authorize :authenticate} type)]}
  (let [authorize-uri (case type
                        :authorize "https://api.twitter.com/oauth/authorize"
                        :authenticate "https://api.twitter.com/oauth/authenticate")]
    (oauth/make-consumer (env :twitter-consumer-token)
                         (env :twitter-consumer-secret)
                         "https://api.twitter.com/oauth/request_token"
                         "https://api.twitter.com/oauth/access_token"
                         authorize-uri
                         :hmac-sha1)))

(defn authorize [{headers :headers session :session params :params :as req}]
  (let [consumer (make-consumer :type (:type params))
        request-token (oauth/request-token consumer (helpers/derive-callback headers "/twitter/callback"))]
    {:status 200
     :session (assoc session :twitter {:consumer consumer :request-token request-token}) 
     :body (oauth/user-approval-uri consumer (:oauth_token request-token) {:force_login "true"})}))

(defn authenticate [{{query :params {{request-token :request-token consumer :consumer} :twitter} :session} :request}]
  (if (contains? query :denied)
    [false {::error "Authentication was denied."}]
    (try 
      [true {::access-token-response (oauth/access-token consumer
                                                         request-token
                                                         (query :oauth_verifier))}]
      (catch Exception e [false {::error (ex-data e)}]))))

(defresource callback
  :available-media-types ["text/html"]
  :authorized? (fn [ctx]
                 (authenticate ctx))
  :handle-ok (fn [{{session :session} :request :as ctx}]
               (-> (util/redirect "/")
                   (assoc :session (assoc session :twitter (get-in ctx [::access-token-response])))
                   (assoc :flash {:message "Successfully signed in via Twitter." :level :success})
                   (ring-response)))
  :handle-unauthorized (fn [{{session :session} :request :as ctx}]
                         (let [session (dissoc session :twitter)]
                           (-> (util/redirect "/")
                               (assoc :session {:authentication-error (get-in ctx [::error])})
                               (ring-response)))))
