(ns facetious.authentication.google
  (:require [gapi.auth :as auth]
            [gapi.core :refer [call build]]
            [liberator.core :refer [defresource log!]]
            [environ.core :refer [env]]
            [liberator.representation :refer [ring-response as-response]]
            [ring.util.response :as util]))

(def ^{:dynamic true :doc "Dynamic var used over the life of a Ring session."} *auth*)
(def scopes ["https://www.googleapis.com/auth/plus.me" "https://www.googleapis.com/auth/userinfo.email" "https://www.googleapis.com/auth/youtube"])
(def service (build "https://www.googleapis.com/discovery/v1/apis/plus/v1/rest"))
(def oauth-service (build "https://www.googleapis.com/discovery/v1/apis/oauth2/v2/rest"))

(defn authorize [{session :session :as req}]
  (binding [*auth* (auth/create-auth (env :client-id) (env :client-secret) (env :redirect-url))]
    (let [url (auth/generate-auth-url *auth* scopes)
          auth {:auth *auth* :url url}]
      {:status 200 :session (assoc session :google auth) :body (:url auth)})))

(declare about-me)
(defn authenticate
  "Authenticate user"
  [ctx]
  (let [query (get-in ctx [:request :params])
        auth (get-in ctx [:request :session :google :auth])]
    (auth/exchange-token auth (:code query) (:state query))
    (auth/is-valid auth)))

(defresource callback
  :available-media-types ["text/html"]
  :authorized? (fn [ctx]
             (authenticate ctx))
  :exists? (fn [ctx]
             (let [me (about-me ctx)
                   auth (get-in ctx [:request :session :google :auth])]
               {::me me}))
  :handle-ok (fn [{me ::me :as ctx}] (-> (util/redirect "/")
                                        (assoc :flash {:message (str (:given_name me) " successfully authorized by Google.") :level :success})
                                        (assoc :session (get-in ctx [:request :session]))
                                        (assoc-in [:session :me] me)
                                        (ring-response))))

(defn token-valid?
  "Checks if google+ token is valid (could implement an auto refresh)"
  [ctx]
  (let [auth (get-in ctx [:request :session :google :auth])]
    (and auth (auth/is-valid auth))))

(defn token-info
  "Checks if google+ token is valid (could implement an auto refresh)"
  [ctx]
  (let [auth (get-in ctx [:request :session :google :auth])]
    (println-str "Token expires in" (float (/ (- (:expires @auth) (System/currentTimeMillis)) 60000)) "minutes")))

(defn about-me
  "What I want from the Google+ profile"
  [ctx]
  (let [auth (get-in ctx [:request :session :google :auth])
        me (call auth service "plus.people/get" {"userId" "me"})
        userinfo (call auth oauth-service "oauth2.userinfo/get" {})]
    (merge me userinfo)))

(defresource auth-info
  :available-media-types ["text/plain"]
  :authorized? (fn [ctx] (token-valid? ctx))
  :handle-unauthorized "This page requires authorization."
  :handle-ok (fn [ctx] 
               (str (token-info ctx) (about-me ctx))))
