(ns facetious.authentication.etsy
  (:require [oauth.client :as oauth]
            [environ.core :refer [env]]
            [liberator.core :refer [defresource log!]]
            [liberator.representation :refer [ring-response as-response]]
            [facetious.helpers :as helpers]
            [ring.util.response :as util]
            [clojure.java.io :as io]))

;; entry point is auth, with type authorize, authenticate or force-login (Twitter sign-in flow).

(defn make-consumer []
  (oauth/make-consumer (env :etsy-consumer-token)
                       (env :etsy-consumer-secret)
                       "https://openapi.etsy.com/v2/oauth/request_token"
                       "https://openapi.etsy.com/v2/oauth/access_token"
                       "https://openapi.etsy.com/v2/oauth/authorize"
                       :hmac-sha1))

(defn authorize [{headers :headers session :session params :params :as req}] 
  (let [consumer (make-consumer)
        request-token (oauth/request-token consumer (helpers/derive-callback headers "/etsy/callback") {:scope "email_r listings_r"})]
    {:status 200
     :session (assoc session :etsy {:consumer consumer :request-token request-token}) 
     :body (:login_url request-token)}))


(defn authenticate [{{query :params {{request-token :request-token consumer :consumer} :etsy} :session} :request}]
  (try 
    [true {::access-token-response (oauth/access-token consumer
                                                       request-token
                                                       (query :oauth_verifier))}]
    (catch Exception e [false {::error (ex-data e)}])))

(defresource callback
  :available-media-types ["text/html"]
  :authorized? (fn [ctx]
             (authenticate ctx))
  :as-response (fn [d {{session :session} :request :as ctx}]
                 (let [session (assoc session :etsy (ctx ::access-token-response))]
                   (-> (as-response d ctx)
                       (assoc :session (dissoc session :request-token :consumer)))))
  :handle-ok (fn [ctx] (-> (util/redirect "/")
                          (assoc :flash {:message "Successfully authorized by Etsy." :level :success})
                          (ring-response))))
