(ns twitter-fu.front-end.webapp.authentication.twitter
  (:require [oauth.client :as oauth]
            [clj-http.client :as httpclient]
            [environ.core :refer [env]]
            [liberator.core :refer [defresource log!]]
            [liberator.representation :refer [ring-response as-response]]
            [ring.util.response :as util]))


;; entry point is auth, with type authorize, authenticate or force-login (Twitter sign-in flow).

(defn make-consumer [& {type :type}]
  {:pre [(contains? #{:authorize :authenticate} type)]}
  (let [authorize-uri (case type
                        :authorize "https://api.twitter.com/oauth/authorize"
                        :authenticate "https://api.twitter.com/oauth/authenticate")]
    (oauth/make-consumer (env :twitter-consumer-token)
                         (env :twitter-consumer-secret)
                         "https://api.twitter.com/oauth/request_token"
                         "https://api.twitter.com/oauth/access_token"
                         authorize-uri
                         :hmac-sha1)))

(defn derive-callback [headers]
  (str "http://" (get headers "host") "/twitter/callback"))

(defn authorize [{headers :headers session :session params :params :as req}]
  (let [consumer (make-consumer :type (:type params))
        request-token (oauth/request-token consumer (derive-callback headers))]
    {:status 200
     :session (assoc session :twitter {:consumer consumer :request-token request-token}) 
     :body (oauth/user-approval-uri consumer (:oauth_token request-token) {:force_login "true"})}))

(defn authenticate [{{query :params {{request-token :request-token consumer :consumer} :twitter} :session} :request}]
  (if (contains? query :denied)
    [true {::denied true}]
    (try 
      [true {::access-token-response (oauth/access-token consumer
                                                         request-token
                                                         (query :oauth_verifier))}]
      (catch Exception e [false {::error (ex-data e)}]))))


(defresource callback
  :available-media-types ["text/html"]
  :authorized? (fn [ctx]
                 (authenticate ctx))
  :as-response (fn [d {{session :session} :request :as ctx}]
                 (condp  #(contains? %2 %1) ctx
                   ::denied (-> (as-response d ctx)
                                (assoc :session (dissoc session :request-token :twitter)))
                   ::access-token-response (let [session (assoc-in session [:twitter] 
                                                                   {:access-token-response (get-in ctx [::access-token-response])})]
                                             (-> (as-response d ctx)
                                                 (assoc :session (dissoc session :request-token :consumer))))))
  :handle-ok (fn [ctx]
               (-> (util/redirect "/")
                   (assoc :flash {:message "Successfully signed up via Twitter." :level :success})
                   (ring-response))))
