(ns cljs-utils.react.error-boundary
  (:require [goog.object :as gobj]))

(defn error-boundary
  "Creates a React Error Boundary using direct JS interop.
   Options:
   - fallback-fn: Function that takes (error, error-info, reset-fn) and returns React element
   - on-error-fn: Function called with error details for logging
   - reset-keys: Vector of values that will reset the boundary when changed"
  [& {:keys [fallback-fn on-error-fn reset-keys]}]
  (let [;; Create the class constructor
        ErrorBoundary
        (fn [props]
          (this-as this
            ;; Call React.Component constructor
            (.call js/React.Component this props)
            ;; Initialize state
            (set! (.-state this) #js {:hasError false
                                      :error nil
                                      :errorInfo nil
                                      :resetKeys reset-keys})
            this))

        ;; Set up prototype chain
        _ (set! (.-prototype ErrorBoundary)
                (js/Object.create (.-prototype js/React.Component)))

        ;; Add static getDerivedStateFromError method
        _ (set! (.-getDerivedStateFromError ErrorBoundary)
                (fn [error]
                  #js {:hasError true :error error}))

        ;; Add componentDidCatch method
        _ (set! (.. ErrorBoundary -prototype -componentDidCatch)
                (fn [error error-info]
                  (this-as this
                    (js/console.error "Error Boundary caught error:" error error-info)
                    ;; Update state with error info
                    (.setState this #js {:errorInfo error-info})

                    ;; Call custom error handler if provided
                    (when on-error-fn
                      (on-error-fn {:error error
                                   :error-info error-info
                                   :component-stack (gobj/get error-info "componentStack")})))))

        ;; Add componentDidUpdate for reset functionality
        _ (set! (.. ErrorBoundary -prototype -componentDidUpdate)
                (fn [prev-props prev-state]
                  (this-as this
                    (let [current-reset-keys (gobj/get (.-state this) "resetKeys")
                          prev-reset-keys (gobj/get prev-state "resetKeys")]
                      ;; Reset if reset keys changed and we're in error state
                      (when (and (gobj/get (.-state this) "hasError")
                                current-reset-keys
                                (not= (str current-reset-keys) (str prev-reset-keys)))
                        (.setState this #js {:hasError false
                                            :error nil
                                            :errorInfo nil}))))))

        ;; Add render method
        _ (set! (.. ErrorBoundary -prototype -render)
                (fn []
                  (this-as this
                    (let [state (.-state this)
                          has-error (gobj/get state "hasError")
                          error (gobj/get state "error")
                          error-info (gobj/get state "errorInfo")
                          children (gobj/get (.-props this) "children")

                          ;; Reset function
                          reset-fn #(.setState this #js {:hasError false
                                                         :error nil
                                                         :errorInfo nil})]

                      (if has-error
                        ;; Render fallback UI
                        (if fallback-fn
                          (fallback-fn error error-info reset-fn)
                          ;; Default fallback UI
                          (js/React.createElement
                            "div"
                            #js {:style #js {:padding "20px"
                                            :border "1px solid #ff6b6b"
                                            :borderRadius "4px"
                                            :backgroundColor "#ffe0e0"
                                            :color "#c92a2a"}}
                            (js/React.createElement "h2" nil "Something went wrong")
                            (js/React.createElement
                              "details" nil
                              (js/React.createElement "summary" nil "Error details")
                              (js/React.createElement
                                "pre"
                                #js {:style #js {:whiteSpace "pre-wrap"
                                                :fontSize "12px"
                                                :marginTop "10px"}}
                                (when error (str error))))
                            (js/React.createElement
                              "button"
                              #js {:onClick reset-fn
                                   :style #js {:marginTop "10px"
                                              :padding "8px 16px"
                                              :backgroundColor "#c92a2a"
                                              :color "white"
                                              :border "none"
                                              :borderRadius "4px"
                                              :cursor "pointer"}}
                              "Try Again")))
                        ;; Render children normally
                        children)))))]

    ;; Return the constructor function
    ErrorBoundary))

(defn create-error-boundary
  "Creates an error boundary React element with children.
   Usage: (create-error-boundary {:fallback-fn my-fallback} child1 child2 ...)"
  [options & children]
  (let [boundary-class (error-boundary options)
        props #js {:children (if (= 1 (count children))
                              (first children)
                              (into-array children))}]
    (js/React.createElement boundary-class props)))

(defn custom-fallback [error error-info reset-fn]
  (js/React.createElement
    "div"
    #js {:className "custom-error"}
    (js/React.createElement "h1" nil "Custom Error UI")
    (js/React.createElement "p" nil "Something went wrong!")
    (js/React.createElement "button" #js {:onClick reset-fn} "Reset")
    (js/React.createElement "pre" nil (str error))))

(defn make-error-boundary-class [default-options]
  (partial error-boundary default-options))
