(ns commos.delta.compscribe
  (:require [commos.delta :as delta]
                  [clojure.core.async :refer [chan close!
                                              <! >!
                                              put!
                                              alts!
                                              go go-loop]]
                                                       
                                                 
                                                
                                                   
            [clojure.walk :refer [prewalk postwalk]])
                                                                       )

(defn- flatten-keys
  "Transform a nested map to a map ks->v where v is not a map."
  [m]
  (let [step (fn step
               [root m]
               (mapcat (fn [[k v]]
                         (if (map? v)
                           (step (conj root k) v)
                           [[(conj root k) v]]))
                       m))]
    (into {} (step [] m))))

(defn- vsplit-at
  "Like split-at, but for vectors."
  [n v]
  [(subvec v 0 n) (subvec v n)])

(defn- group-by-pks
  "Transform a flattened map ks->v into a map pks->[[rks ks v]+] where
  pks is a partial ks and rks the complementary rest.  E. g.:
  {[:foo :bar] 42} becomes {[:foo] ([[:bar] [:foo :bar]
  42]), [] ([[:foo :bar] [:foo :bar] 42])}."
  [m]
  (reduce-kv (fn [acc ks v]
               (reduce (fn [acc [pks rks]]
                         (update acc pks conj [rks ks v]))
                       acc
                       (map #(vsplit-at % ks)
                            (range (count ks)))))
             {}
             m))

(defn- conform-spec
  "Recursively transform spec [endpoint hooks] [endpoint
  flattened-keys-hooks flattened-keys-hooks-grouped-by-pks hooks].
  The resulting structure provides fast lookups required during live
  dispatch."
  [spec]
  (let [mark-spec #(vary-meta % assoc ::spec? true)]
    (->> (mark-spec spec)
         (prewalk (fn [form]
                    (if (::spec? (meta form))
                      (let [[endpoint specs] form]
                        (mark-spec
                         (if (vector? specs)
                           [endpoint {[] (mark-spec specs)}]
                           (let [specs (flatten-keys specs)]
                             [endpoint (zipmap (keys specs)
                                               (map mark-spec
                                                    (vals specs)))]))))
                      form)))
         (postwalk (fn [form]
                     (if (::spec? (meta form))
                       (let [[endpoint specs] form]
                         [endpoint specs (group-by-pks specs)])
                       form))))))

(defn- dissoc-ks
  ;; TODO: This creates a val at ks if it isn't there. A recursive
  ;; algorithm should suit better.
  "Dissoc the last key of ks from (possibly a nested map in) m"
  [m ks]
  (if (= (count ks) 1)
    (dissoc m (peek ks))
    (update-in m (pop ks) dissoc (peek ks))))

(defn- nested-subs
  "Extract necessary subscriptions/unsubscriptions implied by updating
  the new-val at ks.  Returns [subs new-val] where new-val has the
  subscribed ids removed and subs has the extracted
  subscriptions/unsubscriptions merged onto it.

  Pass nil a new-val to only get unsubscriptions."
  [subs deep-hooks ks new-val]
  (reduce (fn [[subs new-val] [rks ks _]]
            (if-let [val (get-in new-val rks)]
              [(if (coll? val)
                 (update-in subs [:subs-many ks] into val)
                 (-> subs
                     (assoc-in [:subs-one ks] val)
                     (update-in [:unsubs] conj ks)))
               (dissoc-ks new-val ks)]
              [(update-in subs [:unsubs] conj ks)
               new-val]))
          [subs new-val]
          (get deep-hooks ks)))

(defn- extract-hooks
  "Return a map of required subscriptions and unsubscriptions with the
  following keys:

  :subs-one ks->val - subscribe hook at ks with val
  :subs-many ks->vals - subscribe hooks at ks with vals
  :unsubs [ks+] - unsubscribe subscriptions at ks

  Subs and unsubs may overlap, unsubs are assumed to be applied
  first."
  [[_ direct-hooks deep-hooks :as conformed-spec] delta]
  (loop [[delta & deltas] (delta/unpack delta)
         subs {}
         adjusted-deltas []]
    (if delta
      (let [[op ks new-val] (delta/diagnostic-delta delta)
            hook (get direct-hooks ks)]
        (case op
          :is (if hook
                (recur deltas
                       (-> subs
                           (assoc-in [:subs-one ks] new-val)
                           (update-in [:unsubs] conj ks))
                       adjusted-deltas)
                (if (associative? new-val)
                  (let [[subs new-val]
                        (nested-subs subs deep-hooks ks new-val)]
                    (recur deltas
                           subs
                           (cond-> adjusted-deltas
                             (seq new-val)
                             (conj
                              (delta/normalized-delta [:is ks new-val])))))
                  (recur deltas
                         subs
                         (conj adjusted-deltas delta))))
          :in (if hook
                (recur deltas
                       (update-in subs [:subs-many ks] into new-val)
                       adjusted-deltas)
                (recur deltas
                       subs
                       (conj adjusted-deltas delta)))
          :ex (if hook
                (recur deltas
                       (update-in subs [:unsubs] into
                                  (map (partial conj ks) new-val))
                       adjusted-deltas)
                (recur deltas
                       (update-in subs [:unsubs] into
                                  (mapcat
                                   (fn [k]
                                     (let [ks (conj ks k)]
                                       (if (contains? direct-hooks ks)
                                         ks
                                         (if-let [nested (get deep-hooks ks)]
                                           (map second nested)))))
                                        new-val))
                       (conj adjusted-deltas delta)))))
      [subs (delta/pack adjusted-deltas)])))

(defprotocol ClosingMix
  ;; Mix in core.async is designed to close only when the target
  ;; channel closes. This leaves no way to determine when all sources
  ;; have been consumed and close the target channel as a
  ;; consequence. This design provides a mix that closes immediately
  ;; when there are no more sources or when the sources have been
  ;; drained.
  (mix-in [this ch])
  (mix-out [this ch]))

(defn- closing-mix
  [target init-chs]
  {:pre [(vector? init-chs)]}
  (let [change (chan)
        channels (atom (conj init-chs))
        m (reify ClosingMix
            (mix-in [_ ch]
              (swap! channels conj ch)
              (put! change true))
            (mix-out [_ ch]
              (swap! channels (comp vec
                                    (partial remove #{ch})))
              (put! change true)))]
    (go-loop []
      (let [chs @channels]
        (if (seq @channels)
          (let [[v ch] (alts! (conj chs change))]
            (cond (identical? change ch)
                  (recur)
                  
                  (nil? v)
                  (do (mix-out m ch)
                      (recur))
                  
                  :else
                  (if (>! target v)
                    (recur))))
          (do
            (close! change)
            (close! target)))))
    m))

(defn- compscribe*
  ;; NOTE: before replacing subs-fn and unsubs-fn with a protocol I want
  ;; to reconsider whether they should return channels
  [outer-target subs-fn unsubs-fn
   [endpoint direct-hooks deep-hooks :as conformed-spec] id]
  (let [;; Once intercepted, events need to go through target-mix so
        ;; that mix-ins and mix-outs have synchronous effects
        target (chan)
        target-mix (closing-mix outer-target [target])
        subs (atom {})
        do-sub (fn [ks id many?]
                 (let [xch (chan 1 (delta/nest (cond-> ks
                                                 many? (conj id))))]
                   (swap! subs assoc ks
                          [xch (compscribe* xch
                                            subs-fn
                                            unsubs-fn
                                            (get direct-hooks ks)
                                            id)])
                   ;; xch will be closed by the subscribed-composition
                   (mix-in target-mix xch)))
        ch-in (subs-fn endpoint id)
        
        transport
        (go-loop []
          (if-some [delta (<! ch-in)]
            (let [[{:keys [subs-one subs-many unsubs]} delta]
                  (extract-hooks conformed-spec delta)]
              
              (doseq [[ks [xch unsubs-fn]] (keep (partial find @subs) unsubs)]
                (unsubs-fn)
                (mix-out target-mix xch)
                (swap! subs dissoc ks))

              (when delta ;; (it is possible that all deltas got eaten up)
                (>! target delta)) ;; Block until evt is put so that
                                   ;; subscriptions are put after and
                                   ;; unsubscriptions are in effect
              
              (doseq [[ks id] subs-one]
                (do-sub ks id false))

              (doseq [[ks ids] subs-many
                      id ids]
                (do-sub ks id true))
              (recur))
            (close! target)))]
    (fn []
      (go
        ;; unsubscribe this:
        (unsubs-fn ch-in)
        
        ;; wait for transport to finish to prevent new subs made by
        ;; stale events
        (<! transport)
        
        ;; unsubs all mixed in transports:
        (doseq [[_ unsubs-fn] (vals @subs)]
          (unsubs-fn))))))

(defn compscribe
  [target subs-fn unsubs-fn spec id]
  (let [conformed-spec (conform-spec spec)]
    (compscribe* target subs-fn unsubs-fn conformed-spec id)))

;;;;;;;;;;;; This file autogenerated from src/cljx/commos/delta/compscribe.cljx
