(ns complete.core
  (:use [clojure.string :only [join]])
  (:require clojure.repl)
  (:import [java.util.jar JarFile] [java.io File StringWriter]
           [java.lang.reflect Method Field Member Modifier]))

;; Code adapted from swank-clojure (http://github.com/jochu/swank-clojure)

(defn namespaces
  "Returns a list of potential namespace completions for a given namespace"
  [ns]
  (map name (concat (map ns-name (all-ns)) (keys (ns-aliases ns)))))

(defn ns-public-vars
  "Returns a list of potential public var name completions for a given namespace"
  [ns]
  (map name (keys (ns-publics ns))))

(defn ns-vars-and-classes
  "Returns a list of all potential var and class name completions for
  a given namespace."
  [ns]
  (map name (keys (ns-map ns))))

(defn ns-classes-qualified
  "Returns a list of potential qualified class name completions for a
  given namespace."
  [ns]
  (for [[_ ^Class val] (ns-map ns) :when (class? val)]
    (.getName val)))

(def special-forms
  (map name '[def if do let quote var fn loop recur throw try monitor-enter monitor-exit dot new set!]))

(defn- static? [^Member member]
  (java.lang.reflect.Modifier/isStatic (.getModifiers member)))

(def ns-java-methods-cache (atom {}))

(defn make-java-methods-cache [ns]
  (loop [cache (transient {})
         [^Member c & r] (for [^Class class (vals (ns-map ns))
                               :when (class? class)
                               ^Member member (concat (.getMethods class) (.getFields class))
                               :when (and (not (static? member)))]
                           (let [dc (.getDeclaringClass member)]
                             (if (= dc class)
                               member
                               (if (instance? Method member)
                                 (.getMethod dc (.getName member (.getParameterTypes ^Method member))
                                 (.getField dc (.getName member)))))))]
    (if c
      (let [full-name (.getName c)]
        (if (cache full-name)
         (recur (assoc! cache full-name (conj (cache (.getName c)) c)) r)
         (recur (assoc! cache full-name [c]) r)))
      (persistent! cache))))

(defn ns-java-methods
  "Returns a list of potential java non-static method name completions
  for a given namespace."
  [ns]
  (let [imported-cls-cnt (count (filter class? (vals (ns-map ns))))]
    (when (or (nil? (@ns-java-methods-cache ns))
              (not= (get-in @ns-java-methods-cache [ns :classes-cnt]) imported-cls-cnt))
      (swap! ns-java-methods-cache
             assoc ns {:classes-cnt imported-cls-cnt
                       :methods (make-java-methods-cache ns)})))
  (for [^Class class (vals (ns-map ns))
        :when (class? class)
        ^Method method (.getMethods class)
        :when (not (static? method))]
    (str "." (.getName method))))

(def static-members-cache (atom {}))

(defn make-static-members-cache [^Class class]
  (loop [cache {}, [^Member c & r] (concat (.getMethods class)
                                           (.getDeclaredFields class))]
    (if c
      (if (static? c)
        (let [full-name (.getName c)]
         (if (cache (.getName c))
           (recur (update-in cache [full-name] conj c) r)
           (recur (assoc cache full-name [c]) r)))
        (recur cache r))
      cache)))

(defn static-members
  "Returns a list of potential static members for a given class"
  [^Class class]
  (when-not (@static-members-cache class)
    (swap! static-members-cache assoc class (make-static-members-cache class)))
  (for [^Member member (concat (.getMethods class) (.getDeclaredFields class))
        :when (static? member)]
    (.getName member)))

;; Documentation extraction. Mostly copied from clojure.repl.

(defn- type-to-pretty-string [^Class t]
  (if (or (.isLocalClass t)
          (.isMemberClass t))
    (.getName t)
    (.getSimpleName t)))

(defn- doc-method-parameters [parameters]
  (->> parameters
       (map type-to-pretty-string)
       (interpose " ")
       join
       (format "(%s)")))

(defn get-member-doc [members]
  (->> members
       (group-by (fn [^Member m] (.getDeclaringClass m)))
       (map (fn [[^Class class, members]]
              (let [^Member f-mem (first members)]
                (str (.getName class) "." (.getName f-mem)
                     (if (instance? Field f-mem)
                       (str " = " (try (.get ^Field f-mem nil)
                                       (catch Exception e "?"))
                            " (" (type-to-pretty-string (.getType ^Field f-mem)) ")\n"
                            (Modifier/toString (.getModifiers f-mem)))
                       (join
                        (map (fn [^Method member]
                               (when (instance? Method member)
                                 (str "\n  " (doc-method-parameters (.getParameterTypes member))
                                      " -> " (type-to-pretty-string (.getReturnType ^Method member))
                                      " (" (Modifier/toString (.getModifiers member)) ")")))
                             (distinct members))))
                     "\n"))))
       (interpose "\n")
       join))

(defn get-static-member-doc [member-str class]
  (get-member-doc (get-in @static-members-cache [class member-str])))

(defn get-ns-java-method-doc [member-str]
  (get-member-doc (get-in @ns-java-methods-cache [*ns* :methods member-str])))

(defn generate-docstring [m]
  (binding [*out* (StringWriter.)]
    (println (str (when-let [ns (:ns m)] (str (ns-name ns) "/")) (:name m)))
    (cond
     (:forms m) (doseq [f (:forms m)]
                  (print "  ")
                  (prn f))
     (:arglists m) (prn (:arglists m)))
    (if (:special-form m)
      (do
        (println "Special Form")
        (println " " (:doc m))
        (if (contains? m :url)
          (when (:url m)
            (println (str "\n  Please see http://clojure.org/" (:url m))))
          (println (str "\n  Please see http://clojure.org/special_forms#"
                        (:name m)))))
      (do
        (when (:macro m)
          (println "Macro"))
        (println " " (:doc m))))
    (str *out*)))

(defn get-doc
  "Prints documentation for a var or special form given its symbol."
  [sym]
  (if-let [special-name ('{& fn catch try finally try} sym)]
    (generate-docstring (#'clojure.repl/special-doc special-name))
    (cond
     (#'clojure.repl/special-doc-map sym)
     (generate-docstring (#'clojure.repl/special-doc sym))

     (find-ns sym)
     (generate-docstring (#'clojure.repl/namespace-doc (find-ns sym)))

     (and (namespace sym) (class? (resolve (symbol (namespace sym)))))
     (get-static-member-doc (name sym) (resolve (symbol (namespace sym))))

     (.startsWith ^String (name sym) ".")
     (get-ns-java-method-doc (subs (name sym) 1))

     (try (meta (resolve sym)) (catch Exception e nil))
     (generate-docstring (meta (resolve sym))))))

(defn resolve-class [sym ns]
  (try (let [val (ns-resolve ns sym)]
         (when (class? val) val))
       (catch Exception e
         (when (not= ClassNotFoundException
                     (class (clojure.main/repl-exception e)))
           (throw e)))))

(defmulti potential-completions
  (fn [^String prefix, ns]
    (cond (.startsWith prefix ".") :method
          (.contains prefix "/")   :scoped
          (.contains prefix ".")   :namespace-or-class
          :else                    :var)))

(defmethod potential-completions :scoped
  [^String prefix, ns]
  (let [scope (symbol (first (.split prefix "/")))]
    (map #(str scope "/" %)
         (if-let [class (resolve-class scope ns)]
           (static-members class)
           (when-let [ns (or (find-ns scope) (scope (ns-aliases ns)))]
             (ns-public-vars ns))))))

(defmethod potential-completions :method
  [_ ns]
  (ns-java-methods ns))

(defmethod potential-completions :namespace-or-class
  [^String prefix, ns]
  (concat (namespaces ns)
          (ns-classes-qualified ns)))

(defmethod potential-completions :var
  [_ ns]
  (concat special-forms
          (namespaces ns)
          (ns-vars-and-classes ns)))

(defn completions
  "Return a sequence of matching completions given a prefix string and an optional current namespace."
  ([prefix] (completions prefix *ns*))
  ([prefix ns]
     (for [^String completion (potential-completions prefix ns)
           :when (.startsWith completion prefix)]
       completion)))
