(ns grenada.guten-tag.more
  "The main feature of this namespace is the deftag+ macro, which adds some
  functionality on top of guten-tag.core/deftag."
  (:require [clojure.string :as string]
            [guten-tag.core :as gt]
            [grenada.schemas :as schemas]))

;;;; Universal helpers

(defn- sym
  "Stringifies and concatenates its arguments, makes an unqualified symbol out
  of the result."
  [& args]
  (symbol (apply str args)))

(defn- sym-name
  "Returns the name part of a symbol as a symbol."
  [kw-or-sym]
  (-> kw-or-sym name symbol))

(defn make-doc
  "Creates doc strings for forms generated by form generators.

  When you write a macro that generates code you want to provide documentation
  both to the user of the macro and to the user of the generated code. Neither
  of those should be forced to look into source code. All documentation should
  be able to documentation page generators. The macro author, Elvira the Lazy,
  doesn't want to write very similar documentation in two places, though. This
  procedure helps her solve that problem.

  When Elvira writes code-generating macros she doesn't want to have all the
  generating code in the body of one macro definition. Instead, she writes a
  separate function G for each defn or def form F to be generated. This is a 1:1
  correspondence. G is of interest to the user of the macro and F is of interest
  to the user of the generated code.

  As said before, Elvira doesn't want to write the same documentation for G and
  F. So she attaches a doc string only to G. The doc string contains some simple
  markup as described below. She then generates the doc string for F by
  executing make-doc on the var holding G _inside_ G.

  Less epic description:

  Reads the doc string of FORM-GENERATOR-VAR. Removes all occurences of '||| …
  |||' and the leading and trailing whitespace that might result. Substitutes
  placeholders according to SUBST-FOR.

  Placholders work like this: if SUBST-FOR contains an entry [:KW 'subst'], all
  occurrences of $KW in the doc string are replaced with 'subst' (excluding
  quotes)."
  [form-generator-var subst-for]
  (let [without-barred-part
        (-> form-generator-var
            meta
            (get :doc "")
            (string/replace #"(?xms)\|{3} .*? \|{3}" "")
            string/trim)]
    (reduce (fn [cur-doc [placeholder replacement]]
              (string/replace cur-doc (str "$" (name placeholder))
                                      (str replacement)))
            without-barred-part
            subst-for)))


;;;; Form generators for the main macro

(defn- t+?-form
  "||| Returns the form for defining a function with this doc string: |||

  Checks whether X is a tagged value of tag type $NAME-SYM. Throws an exception
  if X doesn't adhere to the schema for values of tag type $NAME-SYM."
  [name-sym schema]
  (let [t?-sym (sym name-sym "?")
        doc-string (make-doc #'t+?-form {:NAME-SYM name-sym})]
    `(defn ~(sym t?-sym "+")
       ~doc-string
       [~'x]
       (when (~t?-sym ~'x)
         (s/validate ~schema (gt/val ~'x))
         true))))

(defn- map->t-form
  "||| Returns the form for defining a function with this doc string: |||

  Checks whether M is a map adhering to the schema of tag type $NAME-SYM. If so,
  constructs a tagged value of type $NAME-SYM with this map. Otherwise throws an
  exception."
  [name-sym nmsp-str schema defaults]
  (let [doc-string (make-doc #'map->t-form {:NAME-SYM name-sym})]
    `(defn ~(sym "map->" name-sym)
       ~doc-string
       [~'m]
       (let [m-with-defaults# (s/validate ~schema
                                          (merge ~defaults ~'m))]
         (gt/->ATaggedVal (keyword ~nmsp-str
                                   (str '~name-sym))
                          m-with-defaults#)))))

;; TODO: The distribution of documentation is not optimal here. Think about how
;;       to make it better. (RM  2015-07-21)
(defmacro deftag+
  "Extension to guten-tag.core/deftag.

  Defines the same functions as deftag. Additionally defines functions as
  described in the doc strings doc strings of t+?-form and map->t-form."
  {:grenada.cmeta/bars {:voyt.bars/requires ['schema.core
                                             'grenada.things.schemas
                                             'guten-tag.core]
                        :voyt.bars/defines ["${NAME-SYM}+?"
                                            "map->${NAME-SYM}"
                                            'guten-tag.core/deftag]}}
  ([name-sym fields schema]
   `(deftag+ ~name-sym ~fields ~schema {}))
  ([name-sym fields schema defaults]
   `(do
      (gt/deftag ~name-sym ~fields)
      ~(t+?-form name-sym schema)
      ~(map->t-form name-sym (str (ns-name *ns*)) schema defaults))))
