/*
 * Decompiled with CFR 0.152.
 */
package com.github.swrirobotics.bags.reader.records;

import com.fasterxml.jackson.databind.util.ByteBufferBackedInputStream;
import com.github.swrirobotics.bags.reader.exceptions.BagReaderException;
import com.github.swrirobotics.bags.reader.records.Header;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.SeekableByteChannel;
import net.jpountz.lz4.LZ4FrameInputStream;
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream;
import org.apache.commons.compress.utils.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Record {
    private int headerLength;
    private Header myHeader;
    private Header myConnectionHeader = null;
    private long myDataOffset = -1L;
    private int myDataLength;
    private ByteBuffer myData = null;
    private final SeekableByteChannel myChannel;
    private static final Logger myLogger = LoggerFactory.getLogger(Record.class);

    public Record(SeekableByteChannel channel) throws BagReaderException {
        this.myChannel = channel;
        ByteBuffer lengthBuffer = ByteBuffer.allocate(4);
        try {
            int count = this.myChannel.read(lengthBuffer);
            if (count != 4) {
                throw new BagReaderException("Unable to read header length; only got " + count + " bytes.");
            }
            lengthBuffer.flip();
            this.headerLength = lengthBuffer.order(ByteOrder.LITTLE_ENDIAN).getInt();
            if ((long)this.headerLength > 100000L) {
                throw new BagReaderException("Header is unreasonably large (" + this.headerLength + " bytes).  Bag file may need to be reindexed.");
            }
            if (this.headerLength == 0) {
                myLogger.warn("Saw a zero-byte header.  Skipping.");
                this.myChannel.position(this.myChannel.position() + 4L);
                this.myHeader = new Header();
                return;
            }
            ByteBuffer headerBuffer = ByteBuffer.allocate(this.headerLength);
            count = this.myChannel.read(headerBuffer);
            if (count != this.headerLength) {
                throw new BagReaderException("Unable to read header (size " + this.headerLength + "); only got " + count + " bytes.");
            }
            headerBuffer.flip();
            this.myHeader = new Header(headerBuffer);
            lengthBuffer.flip();
            count = this.myChannel.read(lengthBuffer);
            if (count != 4) {
                throw new BagReaderException("Unable to read data length; only got " + count + " bytes.");
            }
            lengthBuffer.flip();
            this.myDataLength = lengthBuffer.order(ByteOrder.LITTLE_ENDIAN).getInt();
            this.myDataOffset = this.myChannel.position();
            channel.position(channel.position() + (long)this.myDataLength);
        }
        catch (NegativeArraySizeException e) {
            throw new BagReaderException("Unable to read header (size " + this.headerLength + ")");
        }
        catch (IOException e) {
            throw new BagReaderException("Unable to read header.");
        }
    }

    public void readData() throws BagReaderException {
        this.myData = ByteBuffer.allocate(this.myDataLength);
        try {
            this.myChannel.position(this.myDataOffset);
            this.myChannel.read(this.myData);
            if (this.myHeader.getType() == RecordType.CHUNK) {
                String compression;
                switch (compression = this.myHeader.getValue("compression")) {
                    case "none": {
                        break;
                    }
                    case "bz2": 
                    case "lz4": {
                        int decompressedSize = this.myHeader.getInt("size");
                        this.myData.flip();
                        try (ByteBufferBackedInputStream inStream = new ByteBufferBackedInputStream(this.myData);
                             InputStream compressedStream = this.openCompressedStream(compression, (InputStream)inStream);){
                            byte[] buffer = new byte[decompressedSize];
                            int n = IOUtils.readFully((InputStream)compressedStream, (byte[])buffer);
                            if (n != decompressedSize) {
                                throw new BagReaderException("Read " + n + " bytes from a compressed chunk but expected " + decompressedSize + ".");
                            }
                            this.myData = ByteBuffer.wrap(buffer);
                            break;
                        }
                    }
                    default: {
                        myLogger.warn("Unknown compression format: " + compression);
                    }
                }
            }
            this.myData.order(ByteOrder.LITTLE_ENDIAN);
        }
        catch (IOException e) {
            throw new BagReaderException(e);
        }
    }

    private InputStream openCompressedStream(String compressionType, InputStream inStream) throws IOException, BagReaderException {
        switch (compressionType) {
            case "bz2": {
                return new BZip2CompressorInputStream(inStream);
            }
            case "lz4": {
                return new LZ4FrameInputStream(inStream);
            }
        }
        String error = "Unknown compression type: " + compressionType;
        throw new BagReaderException(error);
    }

    public Header getConnectionHeader() {
        return this.myConnectionHeader;
    }

    public void setConnectionHeader(Header connectionHeader) {
        this.myConnectionHeader = connectionHeader;
    }

    public Header getHeader() {
        return this.myHeader;
    }

    public ByteBuffer getData() throws BagReaderException {
        if (this.myData == null) {
            this.readData();
        }
        this.myData.position(0);
        return this.myData;
    }

    public static enum RecordType {
        BAG_HEADER,
        CHUNK,
        CONNECTION,
        MESSAGE_DATA,
        INDEX_DATA,
        CHUNK_INFO,
        UNKNOWN;

    }
}

