(ns boot-javac-star.core
  {:boot/export-tasks true}
  (:require [boot.core :as core :refer [deftask]]
            [boot.util :as util]
            [boot.pod :as pod]
            [boot.file :as file]
            [boot.task.built-in :as built-in]
            boot.repl
            [clojure.set :as set]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [clojure.java.shell :as sh]
            [clojure.pprint :refer [cl-format]]
            [clojure.walk :as walk]
            [boot-javac-star.javac :as javac*])
  (:import (java.util Arrays)
           (javax.tools DiagnosticCollector
                        Diagnostic$Kind)))

(core/deftask javac*
  "Compile java sources."
  [o options OPTIONS [str] "List of options passed to the java compiler."]
  (let [tgt (core/tmp-dir!)]
    (core/with-pre-wrap [fs]
      (let [throw?    (atom nil)
            diag-coll (DiagnosticCollector.)
            opts      (->> ["-d"  (.getPath tgt)
                            "-cp" (System/getProperty "boot.class.path")]
                           (concat options)
                           (into-array String) Arrays/asList)
            already-defined (atom #{})
            handler   {Diagnostic$Kind/ERROR util/fail
                       Diagnostic$Kind/WARNING util/warn
                       Diagnostic$Kind/MANDATORY_WARNING util/warn}
            tmp-srcs  (some->> (core/input-files fs)
                               (core/by-ext [".java"]))]
        (when (seq tmp-srcs)
          (util/info "Compiling %d Java source files...\n" (count tmp-srcs))
          (doseq [tmp-src tmp-srcs]
            (let [prefix (str (core/tmp-dir tmp-src))
                  file (core/tmp-file tmp-src)
                  class-name (javac*/construct-class-name prefix (str file))]
              (javac*/compile-java opts diag-coll class-name (slurp file) already-defined)))
          (doseq [d (.getDiagnostics diag-coll) :let [k (.getKind d)]]
            (when (= Diagnostic$Kind/ERROR k) (reset! throw? true))
            (let [log (handler k util/info)]
              (if (nil? (.getSource d))
                (log "%s: %s\n"
                     (.toString k)
                     (.getMessage d nil))
                (log "%s: %s, line %d: %s\n"
                     (.toString k)
                     (.. d getSource getName)
                     (.getLineNumber d)
                     (.getMessage d nil)))))
          (when @throw? (throw (Exception. "java compiler error")))))
      (-> fs (core/add-resource tgt) core/commit!))))
