(ns org.bituf.sqlrat.clause
  "Convenience functions for handling SQL clauses. Clauses contain
   1. clause expression
   2. clause parameters (optional)
   Clause examples:
   Example 1 -- [\" WHERE emp.id = ?\" 1039]
   Example 2 -- [\" GROUP BY ?\" \"category\"]
   Example 3 -- [\" LIMIT ?, ?\" 50 60]
   Example 4 -- [\"SELECT * FROM emp\"]
   Function names in uppercase (AND, OR, FN) enclose the emitted SQL."
  (:use org.bituf.sqlrat.util)
  (:use org.bituf.sqlrat.clause.internal)
  (:import [clojure.lang IFn IPersistentCollection]))

(def *debug* false)
(defn debug
  ([source message]
    (if *debug* (apply println "[DEBUG][" source "]" message)))
  ([source message data & more]
    (if *debug* (apply println "[DEBUG][" source "]" message data
                  "->" (type data) more))))


;; ========= Validate & Convert as clause =========

(defn clause?
  "Return true if specified object is a clause, false otherwise"
  [obj]
  (true? (clause-key (meta obj))))


(defn as-clause
  "Convert given expression into a clause and return it. Throw exception on
  invalid input. A strictly valid clause is a non-empty vector with the first
  element as the string SQL clause followed by optional values for placeholder
  '?' symbols: [\"id=?\" 45]
  The following clauses are permitted:
  Keyword    - :id       => becomes clause => [\"id\"]
  String     - \"id\"    => becomes clause => [\"id\"]
  Clause     - [\"id\"]  => remains clause => [\"id\"]
  Collection - '(\"id\") => becomes clause => [\"id\"]
  Any value  - 479       => becomes clause => [\"479\"]
  You got the drift."
  [clause]
  (debug "as-clause" "Received clause:" clause)
  (if (clause? clause) clause
    (let [clause-vec (if (string? clause) [clause]
                       (if (keyword? clause) [(name clause)]
                         (if (coll? clause) (as-vector clause)
                           [(str clause)])))
          _          (if (or (nil? clause) (empty? clause-vec))
                       (bad-arg! "Clause must not be nil or empty: " clause))
          _          (if (not (string? (first clause-vec)))
                       (bad-arg! "Clause must begin with string: " clause))
          ]
      (assoc-clause-meta clause-vec))))


(defn as-value-clause
  "Converts a value, or a set of values into a value clause. Value clauses look
  like these:
  [\"?, ?, ?\" 10 20 30]
  [\"?\" \"Anthony Pereira\"]
  etc"
  [value]
  (if *assert-args* (try
                      (assert (not (clause? value)))
                      (assert (not (nil?    value)))
                      (catch AssertionError e
                        (bad-arg!
                          "Invalid input: nil and clauses are not allowed"))))
  (let [values (as-vector value)
        _      (if *assert-args* (doseq [each values]
                                   (try
                                     (assert (not (nil?  each)))
                                     (assert (not (coll? each)))
                                     (assert (not (fn?   each)))
                                     (catch AssertionError e
                                       (bad-arg! "Invalid input: nil, collection and function are not allowed as clause elements"
                                         )))))
        vcount (count values)
        cl-str (apply str
                 (interpose ", "
                   (take vcount
                     (repeat \? ))))]
    (as-clause (into [cl-str] values))))


(defn empty-clause
  []
  (as-clause ""))


(defn empty-clause?
  [clause]
  (assert (clause? clause))
  (empty? (first clause)))



;; ======== Merge clauses =========

(defn merge-clauses
  "Merge one or more clauses into one super-clause."
  [& clauses]
  (debug "merge-clauses" "Received clauses:" clauses)
  (let [clauses-vec (as-vector (map #(as-clause %) clauses))
        qexpr (apply str (map #(first %) clauses-vec))
        qparm (flatten (map #(rest %) clauses-vec))]
    (as-clause (into [qexpr] qparm))))


(defn query
  "Merge clauses with interposed space. Useful to construct free-form clauses.
  Example: (query :SELECT :* :FROM :emp :WHERE (=? :id 56))
           => [\"SELECT * FROM emp WHERE id=?\" 56]
  See also: merge-clauses"
  [& clauses]
  (apply merge-clauses (interpose (as-clause " ") clauses)))


(def ^{:doc "Alias for the 'query' function"}
      >> query)


(defn merge-key-clauses
  [prefix k clause & more]
  (let [clauses (into [clause] more)
        sclause (apply merge-clauses clauses)] ; sclause = super-clause
    (if (empty-clause? sclause) (empty-clause)
      (merge-clauses (str-clause-key k prefix)
        sclause))))


;; short hand functions for merging clauses

(defn enclose
  "Enclose a clause with '(' and ')' characters"
  [clause]
  (merge-clauses (as-clause "(")
    clause (as-clause ")")))

;; short hand

(defn SELECT  [clause & more] (apply merge-key-clauses nil :select
                                (into [clause] more)))
(defn FROM    [clause & more] (apply merge-key-clauses " " :from
                                (into [clause] more)))
(defn WHERE   [clause & more] (apply merge-key-clauses " " :where
                                (into [clause] more)))
(defn GROUPBY [clause & more] (apply merge-key-clauses " " :group-by
                                (into [clause] more)))
(def ^{:doc "Alias for GROUPBY"}
      GROUP-BY GROUPBY)


;; ========== WHERE clause functions ===========

;;;
;; Convenience functions to join clauses by a delimiter

(defn enclose-interpose
  [sep-clause clause-1 & more]
  (let [clauses (into [clause-1] more)]
    (enclose (apply merge-clauses
               (interpose sep-clause clauses)))))


(defn AND
  "Interpose clauses with \" AND \" and enclose the result"
  ([clauses]
    (if *assert-args* (assert (coll? clauses)))
    (apply AND clauses))
  ([clause-1 clause-2 & more]
    (apply enclose-interpose " AND " clause-1 clause-2 more)))


(defn OR
  "Interpose clauses with \" OR \" and enclose the result"
  ([clauses]
    (if *assert-args* (assert (coll? clauses)))
    (apply OR clauses))
  ([clause-1 clause-2 & more]
    (apply enclose-interpose " OR " clause-1 clause-2 more)))


;;;
;; functions on 1 operand
(defn is-null
  "Is null
   Example: (is-null :location) => [\"location IS NULL\"]"
  [k]
  (assert k)
  (as-clause [(str (str-name k) " IS NULL")]))


(defn not-null
  "Not null
   Example: (not-null :location) => [\"location NOT NULL\"]"
  [k]
  (assert k)
  (as-clause [(str (str-name k) " NOT NULL")]))


(defn is-not-null
  "Is not null
   Example: (is-not-null :location) => [\"location IS NOT NULL\"]"
  [k]
  (assert k)
  (as-clause [(str (str-name k) " IS NOT NULL")]))


;;;
;; functions on 2 operands: ?-suffixed function names indicate parameterized SQL
;; taken from here: http://com.w3schools.com/sql/sql_where.asp
(defn op2
  "Join column and value by specified operator. Use to write helper functions.
   Note: 'op' and 'k' are asserted not to be a logical false in 3-arg version.
   Example: (op2 \\= :id 45)
            => [\"id=?\" 45]
   Example: (op2 \\= :id (as-clause \"SELECT x FROM y WHERE z=10\"))
            => [\"id=(SELECT x FROM y WHERE z=10)\"]
   Example: (op2 \\= :id nil (fn [k] (str k \" IS NULL\")))
            => [\":id IS NULL\"]
   Example: (op2 \\= :id 697 (fn [k] (str k \" IS NULL\")))
            => [\"id=?\" 697]"
  ([op k v]
    (assert op)
    (assert k)
    (if (clause? v) (merge-clauses (as-clause (str (str-name k) op))
                      (enclose v)) ; sub-query
      (as-clause [(str (str-name k) op \?) v])))
  ([op k v if-nil-fn1]
    (if (nil? v) (if-nil-fn1 k)
      (op2 op k v))))


(defn =?
  "Equals
   Example: (=? :id 45) >> [\"id=?\" 45]"
  [k v]
  (op2 \=   k v is-null))


(defn <>?
  "Not equal to
   Example: (<>? :id 56) >> [\"id<>?\" 56]"
  [k v]
  (op2 "<>" k v not-null))


(defn !=?
  "Not equal to
   Example: (!=? :id 56) >> [\"id!=?\" 56]"
  [k v]
  (op2 "!=" k v not-null))


(defn >?
  "Greater than
   Example: (>? :c 8.3) >> [\"c>?\" 8.3]"
  [k v]
  (assert v)
  (op2 \>   k v))


(defn <?
  "Less than
   Example: (<? :qty 5) >> [\"qty<?\" 5]"
  [k v]
  (assert v)
  (op2 \<   k v))


(defn >=?
  "Greater than or equals
   Example: (>=? :t 9.6) >> [\"t>=?\" 9.6]"
  [k v]
  (assert v)
  (op2 ">=" k v))


(def =>? >=?)


(defn <=?
  "Less than or equals
   Example; (<=? :t 9.7) >> [\"t<=?\" 9.7]"
  [k v]
  (assert v)
  (op2 "<=" k v))


(def =<? <=?)


(defn like?
  "Like
   Example: (like? :flag \"on\") >> [\"flag LIKE ?\" \"on\"]"
  [k v]
  (assert v)
  (op2 " LIKE " k v))


(defn begins-with?
  "Begins with
   Example: (begins-with? :name \"ram\") >> [\"name LIKE ?\" \"ram%\"]"
  [k v]
  (assert v)
  (like? k (str v \%)))


(defn ends-with?
  "Ends with
   Example: (ends-with? :name \"reynolds\") >> [\"name LIKE ?\" \"%reynolds\"]"
  [k v]
  (assert v)
  (like? k (str \% v)))


(defn includes?
  "Includes
   Example: (includes? :name \"matt\") >> [\"name LIKE ?\" \"%matt%\"]"
  [k v]
  (assert v)
  (like? k (str \% v \%)))


;;;
;; BETWEEN expression
(defn between?
  "Value between v1 and v2.
   Example: (between? :p 6 9) >> [\"p BETWEEN ? AND ?\" 6 9]"
  [k v1 v2]
  (assert k)
  (assert v1)
  (assert v2)
  (as-clause [(str (str-name k) " BETWEEN ? AND ?") v1 v2]))


;;;
;; IN expression
(defn- in-g?
  "Generic IN expression function" 
  [^String kw k v-coll]
  (assert k)
  (assert v-coll)
  (if (clause? v-coll) (merge-clauses (str-name k) kw
                         (enclose v-coll)) ; sub-query
    (let [v-vec (as-vector v-coll)
          _     (if (empty? v-vec)
                  (throw (IllegalArgumentException. "Value collection is empty")))
          v-clauses (map #(if (clause? %) (enclose %) ; deep sub-query
                            (as-value-clause %)) v-vec)]
      (merge-clauses (as-clause k) kw
        (apply enclose-interpose ", " v-clauses)))))


(defn in?
  "In
   Example: (in? :c [30 38]) >> [\"c IN (?, ?)\" 30 38]"
  [k v-coll]
  (in-g? " IN " k v-coll))


(defn not-in?
  "Not in
   Example: (not-in? :c [30 38]) >> [\"c NOT IN (?, ?)\" 30 38]"
  [k v-coll]
  (in-g? " NOT IN " k v-coll))


;;;
;; IN/Equals expression
(defn in=?
  "Either equals or in a set of values.
   Example: (in=? :qty nil) >> [\"qty IS NULL\"]
   Example: (in=? :qty 56) >> [\"qty=?\" 56]
   Example: (in=? :qty [56 78]) >> [\"qty IN (?, ?)\" 56 78]"
  [k v]
  (if (coll? v)
    (in? k v) (=? k v)))


;;;
;; NOT IN/DoesNotEqual expression
(defn- not-in<>-g?
  "Generic 'Neither equals nor in a set of values' function"
  [^IFn ne-fn k v]
  (if (coll? v)
    (not-in? k v) (ne-fn k v)))


(defn not-in<>?
  "Not equal, nor in a set of values.
   Example: (not-in<>? :loc nil)
            >> [\"loc NOT NULL\"]
   Example: (not-in<>? :loc \"egypt\")
            >> [\"loc<>?\" \"egypt\"]
   Example: (not-in<>? :loc [\"egypt\" \"russia\"])
            >> [\"loc NOT IN (?, ?)\" \"egypt\" \"russia\"]"
  [k v]
  (not-in<>-g? <>? k v))


(defn not-in!=?
  "Not equal, nor in a set of values.
   Same as 'not-in<>?' but uses '!=' as operator."
  [k v]
  (not-in<>-g? !=? k v))


;;;
;; Convenience functions for parameters in a map
(defn map-to-clauses
  "Apply function op2fn to a map. Example:
   user=> (map-to-clauses in=? {:a 10 :b [20 30] :c nil})
   ([\"a=?\" 10] [\"b IN (?, ?)\" 20 30] [\"c IS NULL\"])
   The function op2fn should take 2 operands and must return a clause."
  [op2fn kvmap]
  (map #(op2fn (first %) (last %)) (seq kvmap)))


(defn ?
  "Smart value-parameter function. If you pass a map, a key=value pattern is
  applied. For other collections and single-value a parameterized value clause
  is generated. 
  Arguments:
    coll  (Collection) of data. If it is a map then equivalent to (? in=? coll)
          else generates a parameterized SQL value clause
    op2fn (Function) accepts two arguments, returns parameterized SQL clause
    m     (Map) map of colnames versus values
  Example:
    user=> (? [1 2 3])
    [\"?, ?, ?\" 1 2 3]
    user=> (? #{10 nil 55})
    [\"?, ?, ?\" nil 10 55]
    user=> (? 34)
    [\"?\" 34]
    user=> (? {:a 10 :b \"hello\" :c [10 20 30] :d nil})
    ([\"a=?\" 10] [\"b=?\" \"hello\"]
     [\"c IN (?, ?, ?)\" 10 20 30] [\"d IS NULL\"])
  See also:
    as-value-clause, map-to-clauses"
  ([value]
    (if (map? value) (? in=? value)
      (as-value-clause value)))
  ([op2fn m]
    (if *assert-args*
      (assert (fn? op2fn))
      (assert (map? m)))
    (map-to-clauses op2fn m)))


;; === Comma separated names (for SELECT columns, GROUP BY, ORDER BY etc) ===

(defn csnames
  "Return columns in comma separated string form.
   Example: [:qty :price \"order_date\"] >> [\"qty, price, order_date\"] "
  [tokens]
  (if (empty? tokens) (empty-clause)
    (let [tokens-vec (as-vector tokens)]
      (apply merge-clauses
        (interpose (as-clause ", ")
          (map as-clause tokens-vec))))))


(defn >|
  "Flat-style calling format for the 'csnames' function"
  [& tokens] (csnames tokens))


;; === The LIMIT clause ===

(defn limit
  "The LIMIT clause"
  ([howmany]
    (as-clause [" LIMIT ?" howmany]))
  ([from howmany]
    (as-clause [" LIMIT ?, ?" from howmany])))


;; === Functions and Columns ===


(defn sqlfn
  "Example: user=> (sqlfn :AVG :subjectid)
            [\"AVG(subjectid)\"]"
  [fn-name & args]
  (merge-clauses
    (apply merge-clauses
      (as-clause (str (str-name fn-name) "("))
      (map as-clause args))
    (as-clause ")")))

(def ^{:doc "Alias for 'sqlfn' function"}
      FN sqlfn)


(defn as
  "Example: user=> (as :avgscore \"AVG(score)\")
            [\"AVG(score) AS avgscore\"]"
  [new-colname expr]
  (apply merge-clauses
    (map as-clause [expr " AS " new-colname])))

(defn asfn
  "Example: user=> (asfn :avgscore :AVG :score)
            [\"AVG(score) AS avgscore\"]"
  [new-colname fn-name & args]
  (as new-colname (apply sqlfn fn-name args)))
