;;; SPDX-FileCopyrightText: 2024 Jomco B.V.
;;; SPDX-FileCopyrightText: 2024 Topsector Logistiek
;;; SPDX-FileContributor: Joost Diepenmaat <joost@jomco.nl>
;;; SPDX-FileContributor: Remco van 't Veer <remco@jomco.nl>
;;;
;;; SPDX-License-Identifier: AGPL-3.0-or-later

(ns org.bdinetwork.ishare.client
  "Provides an iSHARE-compatible HTTP client.

  The iSHARE client allows for automatic authentication of requests
  and adherence checks of contacted services.

  The client namespace provides `*-request` functions to build HTTP
  requests, and an `exec` function with iSHARE specific middleware to
  execute the created requests.

  ## Configuring

  The client is configured by passing namespaced keys to the request
  builders:

  ```clojure
  (require '[org.bdinetwork.ishare.client :as client])
  (require '[org.bdinetwork.ishare.client.request :as request])

  (def config {:ishare/satellite-base-url \"http://example.com\"
               :ishare/private-key ....
               :ishare/x5c ...
               ...}

  (-> config ;; start from base config
      ;; add specific config for this request
      (assoc :ishare/server-id \"some-server\"
             :ishare/base-url \"https://some.other.example.com\")
      ;; request for standardized \"/capabilities\" endpoint
      (request/capabilities-request)
      ;; execute request
      (client/exec))
  ```

  ## Request builders

  Request builder functions are available in the
  `org.bdinetwork.ishare.client.request` namespace.

  ## Configuration keys

  The following keywords can be provided in requests.

  `:ishare/satellite-base-url` - The URL of the iSHARE Satellite to
  use when obtaining party information.

  `:ishare/satellite-id` - The ID of the iSHARE Satellite to use when
  obtaining party information.

  `:ishare/x5c` - The full certificate chain to use for
  authentication, as a vector of X509 certificates, can be created by
  the `x5c` function.

  `:ishare/private-key` - The private key to use for authentication,
  can be created by the `private-key` function.

  `:ishare/client-id` - The client ID to use for authentication. The
  client ID, certificate chain and private key must match the client's
  registration in the iSHARE Satellite.

  `:ishare/bearer-token` - The access token to use for the current
  request, used by `bearer-token-interceptor`. If not provided, the
  `fetch-bearer-token-interceptor` will attempt to authenticate and
  fetch an access token`.

  `:ishare/base-url` - used by `build-uri-interceptor` to create a
  full `:uri` for the given request, if `:path` is also present. If
  `:uri` is already present, `:ishare/base-url` and `:path` can be
  omitted.

  `:ishare/dataspace-id` - the Dataspace ID to be used for the
  request. This is relevant for `fetch-issuer-ar-interceptor` but may
  be used in other contexts in the future.

  `:ishare/check-server-adherence?` - defaults to `true`, meaning that
  before contacting a server, the server's party info will be
  requested from the Satellite. If the server is not currently
  adherent and active, an exception is raised. If
  `:ishare/check-server-adherence?` is provided and `false`, this
  check is disabled.

  `:ishare/unsign-token` - if provided, specifies a attribute which,
  if present in the response body, has an iSHARE JWT value that should
  be validated and decoded using
  `org.bdinetwork.ishare.jwt/unsign-token`.

  `:ishare/lens` - path to the \"result\" in the response map; the
  object at the path will be placed in the resonse under
  `:ishare/result`

  ## Informational keys

  The following keywords may be added by middleware or request
  builders. If the

  `:ishare/server-name` - added to request by fetching the server info
  from the iSHARE Satellite. Will not be added if
  `:ishare/check-server-adherence?` is `false`.

  `:ishare/server-adherent?` - added to request by fetching the server
  info from the iSHARE Satellite. Will not be added if
  `:ishare/check-server-adherence?` is `false`.

  `:ishare/policy-issuer` - the policy issuer to be used for a
  `delegation-evidence-request`. This is relevant for
  `fetch-issuer-ar-interceptor`

  `:ishare/operation` - marks the request type as generated by the
  `*-request` builders.

  `:ishare/result` - if present in the response, represents \"the
  result\" of executing the request. For standard iSHARE requests,
  this is generally the (decoded) result token.

  ## Interceptors / middleware

  The `org.bdinetwork.ishare.client.interceptors` namespace contains
  the client middleware for processing requests and responses during
  execution."
  (:require [babashka.http-client :as http]
            [babashka.http-client.interceptors :as babashka.interceptors]
            [buddy.core.keys :as keys]
            [clojure.string :as string]
            [org.bdinetwork.ishare.client.interceptors :as ishare.interceptors]))

(defn private-key
  "Read private key from file."
  [key-file]
  (keys/private-key key-file))

;; From https://dev.ishareworks.org/reference/jwt.html#refjwt
;;
;;  "Signed JWTs MUST contain an array of the complete certificate
;;   chain that should be used for validating the JWT’s signature in
;;   the x5c header parameter up until an Issuing CA is listed from
;;   the iSHARE Trusted List."
;;
;; Does this mean we don't need to include the trusted CAs in the x5c
;; chain?

(defn x5c
  "Read `x5c-file` into vector of certificates.

  The `x5c-file` must be the path to a PEM file containing multiple
  X509 certificates."
  [x5c-file]
  (->> (-> x5c-file
           slurp
           (string/replace-first #"(?s)\A.*?-+BEGIN CERTIFICATE-+\s+" "")
           (string/replace #"(?s)\s*-+END CERTIFICATE-+\s*\Z" "")
           (string/split #"(?s)\s*-+END CERTIFICATE-+.*?-+BEGIN CERTIFICATE-+\s*"))
       (mapv #(string/replace % #"\s+" ""))))


(def ^:private interceptors
  [ishare.interceptors/fetch-issuer-ar-interceptor
   ishare.interceptors/ishare-server-adherence-interceptor
   ishare.interceptors/throw-on-exceptional-status-code
   ishare.interceptors/log-interceptor
   ishare.interceptors/logging-interceptor
   ishare.interceptors/lens-interceptor
   ishare.interceptors/unsign-token-interceptor
   ishare.interceptors/build-uri-interceptor
   ishare.interceptors/fetch-bearer-token-interceptor
   ishare.interceptors/bearer-token-interceptor
   babashka.interceptors/construct-uri
   babashka.interceptors/accept-header
   babashka.interceptors/query-params
   babashka.interceptors/form-params
   ishare.interceptors/json-interceptor ;; should be between decode-body and
   ;; throw-on-exceptional-status-code, so that JSON
   ;; error messages are decoded
   babashka.interceptors/decode-body
   babashka.interceptors/decompress-body])

;; Client / exec functions

(def ^:dynamic http-client nil)

(def ^:private timeout-ms 10000)

(def ^:private http-client-opts
  {:follow-redirects :normal
   :connect-timeout  timeout-ms
   :request          {:headers {:accept          "*/*"
                                :accept-encoding ["gzip" "deflate"]
                                :user-agent      "clj-ishare-client"}}})

(def default-http-client
  (delay (http/client http-client-opts)))

(defn exec
  "Execute an iSHARE `request`.

  See `org.bdinetwork.ishare.client` namespace documentation.

  See `org.bdinetwork.ishare.client.request` for request builders."
  [request]
  (http/request (assoc request
                       :client (or http-client @default-http-client)
                       :interceptors interceptors
                       :timeout timeout-ms)))
