(ns babashka.cli.exec
  (:require
   [babashka.cli :as cli :refer [merge-opts parse-opts]]
   [clojure.edn :as edn]
   [clojure.string :as str])
  (:import [java.util.concurrent Executors ThreadFactory]))

(set! *warn-on-reflection* true)

(def ^:private ^:dynamic *basis* "For testing" nil)

(defn- resolve-exec-fn [ns-default exec-fn]
  (if (simple-symbol? exec-fn)
    (symbol (str ns-default) (str exec-fn))
    exec-fn))

(defn- set-daemon-agent-executor
  "Set Clojure's send-off agent executor (also affects futures). This is almost
  an exact rewrite of the Clojure's executor, but the Threads are created as
  daemons.

  From https://github.com/clojure/brew-install/blob/271c2c5dd45ed87eccf7e7844b079355297d0974/src/main/clojure/clojure/run/exec.clj#L171"
  []
  (let [thread-counter (atom 0)
        thread-factory (reify ThreadFactory
                         (newThread [_ runnable]
                           (doto (Thread. runnable)
                             (.setDaemon true) ;; DIFFERENT
                             (.setName (format "CLI-agent-send-off-pool-%d"
                                               (first (swap-vals! thread-counter inc)))))))
        executor (Executors/newCachedThreadPool thread-factory)]
    (set-agent-send-off-executor! executor)))

(defn- parse-exec-opts [args]
  (let [basis (or *basis* (some->> (System/getProperty "clojure.basis")
                                   slurp
                                   (edn/read-string {:default tagged-literal})))
        argmap (or (:argmap basis)
                   ;; older versions of the clojure CLI
                   (:resolve-args basis))
        exec-fn (:exec-fn argmap)
        ns-default (:ns-default argmap)
        first-arg (first args)
        [base-args args] (if (and (not exec-fn)
                                  first-arg
                                  (not (str/includes? first-arg "/")))
                           (let [base-arg-count (if ns-default 1 2)]
                             [(take base-arg-count args) (drop base-arg-count args)])
                           [nil args])
        {:keys [cmds args]} (cli/parse-cmds args)
        cmds (concat base-args cmds)
        [f & cmds] cmds
        [cli-opts cmds] (cond (not f) nil
                              (str/starts-with? f "{")
                              [(edn/read-string f) cmds]
                              :else [nil (cons f cmds)])
        [f unconsumed-args] (case (count cmds)
            0 [(resolve-exec-fn ns-default exec-fn)]
            1 [(let [f (first cmds)]
                 (if (str/includes? f "/")
                   (symbol f)
                   (resolve-exec-fn ns-default (symbol f))))]
            (let [[ns-default f & unconsumed-args] cmds]
              [(if (str/includes? f "/")
                     (symbol f)
                     (resolve-exec-fn (symbol ns-default) (symbol f)))
               unconsumed-args]))
        args (concat unconsumed-args args)
        f* f
        f (requiring-resolve f)
        _ (assert (ifn? f) (str "Could not resolve function: " f*))
        ns-opts (:org.babashka/cli (meta (:ns (meta f))))
        fn-opts (:org.babashka/cli (meta f))
        exec-args (merge-opts
                   (:exec-args cli-opts)
                   (:exec-args argmap))
        opts (merge-opts ns-opts
                         fn-opts
                         cli-opts
                         (:org.babashka/cli argmap)
                         (when exec-args {:exec-args exec-args}))
        opts (parse-opts args opts)]
    [f opts]))

#_(comment
    (System/clearProperty "clojure.basis")
    (binding [*basis* nil]
      (with-redefs [requiring-resolve identity]
        (parse-exec-opts ["dimigi.extraction" "-main" ":a" ":b"])))
    )

(defn main [& args]
  (let [[f opts] (parse-exec-opts args)]
    (set-daemon-agent-executor)
    (f (vary-meta opts assoc-in [:org.babashka/cli :exec] true))))

(defn -main
  "Main entrypoint for command line usage.
  Expects a namespace and var name followed by zero or more key value
  pair arguments that will be parsed and passed to the var. If the
  first argument is map-shaped, it is read as an EDN map containing
  parse instructions.

  Example when used as a clojure CLI alias:
  ``` clojure
  clojure -M:exec clojure.core prn :a 1 :b 2
  ;;=> {:a \"1\" :b \"2\"}
  ```"
  [& args]
  (apply main args))
