(ns sumologic-timbre-appender.core
  (:require [again.core :as again]
            [clojure.pprint :as pprint]
            [slingshot.slingshot :refer [try+ throw+]]))

(defn- log-to-sumo
  [post-fn
   {:keys [max-duration max-retries randomize-numerator multiplier-base
           multiplier-factor]
    :or {max-duration 30000 max-retries 10 randomize-numerator 50
         multiplier-base 500 multiplier-factor 1.5}}
   data]
  (let [{:keys [output_]} data
        output-str (force output_)]
    (try+
      (again/with-retries
        (again/max-duration
          max-duration
          (again/max-retries
            max-retries
            (again/randomize-strategy
              (/ randomize-numerator 100)
              (again/multiplicative-strategy multiplier-base
                                             multiplier-factor))))
        (post-fn output-str))
      (catch Object _
        (binding [*out* *err*]
          (pprint/pprint {:msg "Unable to log to SumoLogic"
                           :output-str output-str
                           :throw-context &throw-context}))))))

(defn sumologic-appender
  "Log some data to SumoLogic. Takes the following arguments:

      post-fn      -- A function which takes a single log statement (string) and
      (required)      posts it to the SumoLogic web service
      options      -- A map containing the following
                      optional keys:

      :max-duration -- The longest we will take to do all
      (default 30000)  our retries
      :max-retries  -- The maximum number of times we will
      (default 10)     retry
      :randomize-numerator -- A seed for a randomly
      (default 50)            generated number
                              divided into 100 that
                              gets added to the retry
                              duration
      :multiplier-base     -- Initial delay to be added to
      (default 500)           the randomized one
      :multiplier-factor -- Number to multiply the previous
      (default 1.5)         number by on every retry"
  [post-fn options]
  {:enabled? true
   :async? true
   :min-level nil
   :rate-limit nil
   :output-fn :inherit
   :fn (partial log-to-sumo post-fn options)})