(ns sumologic-timbre-appender.core
  (:require [again.core :as again]
            [org.httpkit.client :as http]
            [slingshot.slingshot :refer [try+ throw+]]
            [taoensso.timbre :as log]))

(defn- log-to-sumo
  [sumo-url
   {:keys [max-duration max-retries randomize-numerator multiplier-base
           multiplier-factor http-client-options]
    :or {max-duration 30000 max-retries 10 randomize-numerator 50
         multiplier-base 500 multiplier-factor 1.5}}
   data]
  (let [{:keys [output_]} data
        output-str (force output_)]
    (try+
      (again/with-retries
        (again/max-duration
          max-duration
          (again/max-retries
            max-retries
            (again/randomize-strategy
              (/ randomize-numerator 100)
              (again/multiplicative-strategy multiplier-base
                                             multiplier-factor))))
        (let [{:keys [error] :as r}
              @(http/post
                 sumo-url (merge {:body output-str} http-client-options))]
          (log/debugf "SumoLogic request: %s" r)
          (when error (throw+ {:type ::http-error :request r}))))
      (catch Object _
        (log/error (:throwable &throw-context) "Unable to log to SumoLogic")))))

(defn sumologic-appender
  "Log some data to SumoLogic. Takes the following arguments:

      url          -- The URL to hit to submit the log
      (required)      statement
      options      -- A map containing the following
                      optional keys:

      :max-duration -- The longest we will take to do all
      (default 30000)  our retries
      :max-retries  -- The maximum number of times we will
      (default 10)     retry
      :randomize-numerator -- A seed for a randomly
      (default 50)            generated number
                              divided into 100 that
                              gets added to the retry
                              duration
      :multiplier-base     -- Initial delay to be added to
      (default 500)           the randomized one
      :multiplier-factor -- Number to multiply the previous
      (default 1.5)         number by on every retry
      :http-client-options -- A map of options to send to the http-kit client"
  [url options]
  {:enabled? true
   :async? true
   :min-level nil
   :rate-limit nil
   :output-fn :inherit
   :fn (partial log-to-sumo url options)})