(ns request-id.core
  (:require [clojure.data.codec.base64 :as b64]
            [clojure.string :as str]
            [taoensso.timbre :as log])
  (:import (java.nio ByteBuffer)
           (java.util UUID)))

(def default-header-key "x-request-id")

(def ^:dynamic ^:private current-header-name nil)

(defn- byte-str [b] (String. b "UTF-8"))

(defn- add-request-id [data request-id]
  (update data :vargs (partial concat [request-id])))

(defn add-request-id-timbre-middleware [request-id]
  (fn [{:keys [?msg-fmt] :as data}]
    (if ?msg-fmt
      (-> data
          (update :?msg-fmt (partial str "[%s] "))
          (add-request-id request-id))
      (add-request-id data (str "[" request-id "]")))))

(defmacro with-request-id-stamping
  "Given a `request-id`, make sure every log message is stamped with the
  `request-id` in brackets at the beginning"
  [request-id & body]
  `(log/with-merged-config
     {:middleware (conj (:middleware log/*config*)
                        (add-request-id-timbre-middleware ~request-id))}
     ~@body))

(defn url-friendly-guid
  "See <https://www.ietf.org/rfc/rfc3986.txt>"
  []
  (let [uuid (UUID/randomUUID)
        lsb (.getLeastSignificantBits uuid)
        msb (.getMostSignificantBits uuid)
        uuid-bytes (ByteBuffer/wrap (byte-array 16))]
    (.putLong uuid-bytes msb)
    (.putLong uuid-bytes lsb)
    (-> uuid-bytes (.array) b64/encode byte-str
        (str/replace #"=" "") (str/replace #"\+" "_") (str/replace #"/" "~"))))

(defn- assoc-request-id [request-id-header-name r request-id]
  (when r (assoc-in r [:headers request-id-header-name] request-id)))

(defn request-id-header []
  (when-not current-header-name
    (throw (IllegalStateException.
            (str "You may not request the request ID outside of the "
                 "wrap-request-id middleware"))))
  current-header-name)

(defn request-id
  "Given an HTTP request (a la ring), get the current request ID. Must be called
   from within an invocation of the `wrap-request-id` middleware"
  [{:keys [headers]}]
  (get headers (request-id-header)))

(defn wrap-request-id
  "Wrap a handler function in a function that causes a unique request ID to
   be stamped on every log message using a timbre middleware. If you give it
   a `request-id-header-name`, it will grab/generate the header on the request
   and response using the given name. If a value is already present in the
   headers it will use the existing value instead of making a new one."
  ([handler]
   (wrap-request-id default-header-key handler))

  ([request-id-header-name handler]
   (fn [{:keys [headers] :as request}]
     (let [existing-request-id (headers request-id-header-name)
           request-id (or existing-request-id (url-friendly-guid))

           assoc-request-id (partial assoc-request-id request-id-header-name)

           request
           (if existing-request-id
             request
             (assoc-request-id request request-id))]
       (assoc-request-id
        (with-request-id-stamping request-id
          (binding [current-header-name request-id-header-name]
            (handler request)))
        request-id)))))