(ns request-id.core
  (:require [clojure.data.codec.base64 :as b64]
            [clojure.string :as str]
            [taoensso.timbre :as log])
  (:import (java.nio ByteBuffer)
           (java.util UUID)))

(def default-header-key "x-request-id")

(def ^:dynamic current-header-name nil)

(defn- byte-str [b] (String. b "UTF-8"))

(defn- add-request-id [data request-id]
  (update data :vargs (partial concat [request-id])))

(defn- add-request-id-timbre-middleware [request-id]
  (fn [{:keys [?msg-fmt] :as data}]
    (if ?msg-fmt
      (-> data
          (update :?msg-fmt (partial str "[%s] "))
          (add-request-id request-id))
      (add-request-id data (str "[" request-id "]")))))

(defn- url-friendly-guid
  "See <https://www.ietf.org/rfc/rfc3986.txt>"
  []
  (let [uuid (UUID/randomUUID)
        lsb (.getLeastSignificantBits uuid)
        msb (.getMostSignificantBits uuid)
        uuid-bytes (ByteBuffer/wrap (byte-array 16))]
    (.putLong uuid-bytes msb)
    (.putLong uuid-bytes lsb)
    (-> uuid-bytes (.array) b64/encode byte-str
        (str/replace #"=" "") (str/replace #"\+" "_") (str/replace #"/" "~"))))

(defn- assoc-request-id [request-id-header-name r request-id]
  (update r :headers assoc request-id-header-name request-id))

(defn request-id [{:keys [headers]}]
  (when-not current-header-name
    (throw (IllegalStateException.
            (str "You may not request the request ID outside of the "
                 "wrap-request-id middleware"))))

  (get headers current-header-name))

(defn wrap-request-id
  ([handler]
   (wrap-request-id default-header-key handler))

  ([request-id-header-name handler]
   (fn [{:keys [headers] :as request}]
     (let [existing-request-id (headers request-id-header-name)
           request-id (or existing-request-id (url-friendly-guid))

           assoc-request-id (partial assoc-request-id request-id-header-name)

           request
           (if existing-request-id
             request
             (assoc-request-id request request-id))]
       (assoc-request-id
        (log/with-merged-config
         {:middleware [(add-request-id-timbre-middleware request-id)]}
         (binding [current-header-name request-id-header-name]
           (handler request)))
        request-id)))))