(ns piippin.core
  (:require [clojure.string :as str]
            [slingshot.slingshot :refer [throw+]]
            [clojure.walk :as w]))

(def dob (str "(?i)(dob|birth).{0,20}"
              "(\\d{2,4}-\\d{2}-\\d{2}|\\d{2}/\\d{2}/\\d{2,4})"))

(def common-patterns
  {"longnum" (re-pattern "\\d{6,}")
   ;https://www.regular-expressions.info/creditcard.html
   "creditcard" (re-pattern "(?:\\d[ -]*?){13,16}")
   ;http://www.cardinalpath.com/what-you-need-to-know-about-google-analytics-\
   ;personally-identifiable-information/
   "email" (re-pattern (str "([a-zA-Z0-9_\\.-]+)@([\\da-zA-Z\\.-]+)"
                            ".([a-zA-Z\\.]{2,6})"))
   "ssn" (re-pattern "\\d{3}-?\\d{2}-?\\d{4}")
   ;https://stackoverflow.com/questions/16699007/regular-expression-to-match-\
   ;standard-10-digit-phone-number
   "phone" (re-pattern (str "(\\+\\d{1,2}[ \\t])?\\(?\\d{3}\\)?[[ \\t].-]?"
                            "\\d{3}[ \\t.-]?\\d{4}"))
   "dob" (re-pattern dob)})


(defn- simpleid []
  (str/join (map char (take 10 (repeatedly (comp (partial + 97)
                                                 (partial rand-int 26)))))))

(defn replace-with-fingerprint [s replacements]
  {:result (reduce-kv (fn [s k v] (str/replace s k v)) s replacements)
   :replacements replacements})

(defn fingerprint [exclusions s]
  (let [found
        (remove (comp nil? first)
                (map #(vector % (re-seq % s)) exclusions))]
    (replace-with-fingerprint
      s
      (zipmap (map second (reduce concat (map second found)))
              (repeatedly simpleid)))))

(defn return [replacements fingerprinted-string]
  (reduce-kv (fn [s k v] (str/replace s v k))
             fingerprinted-string replacements))

(defn find-pii
  "Given a type `t`, check if the string `s` contains any lines that match
   either the regular expression that is found in `common-patterns` with
   its name or a regular expression that contains the value itself."
  [exceptions t s]
  (let [{:keys [result replacements]} (fingerprint exceptions s)
        lines (str/split-lines result)

        pattern (common-patterns t)
        pattern (or pattern (re-pattern (str "(?i)" t)))]
    (->> lines
         (filter (partial re-find pattern))
         (map (partial return replacements)))))

(defn remove-pii
  "Remove any PII from the given `s`"
  [exceptions s]
  (reduce (fn [s p]
            (let [{:keys [result replacements]} (fingerprint exceptions s)
                  masked (str/replace result p "****")]
              (return replacements masked)))
          (or s "")
          (vals common-patterns)))

(defn remove-pii-from-map
  "Remove PII from a map, provide a [[]] of paths to ignore"
  ([m]
   (remove-pii-from-map m []))
  ([m paths-to-exclude]
   (let [preserved-paths (->> paths-to-exclude
                              (map (fn [path] (hash-map path (get-in m path))))
                              (filter (comp first vals))
                              (apply merge))
         remove-pii (fn [x] (if (or (string? x) (number? x))
                              (remove-pii #{} x)
                              x))
         final-m (reduce-kv (fn [acc k v] (assoc-in acc k v))
                            (w/postwalk remove-pii m)
                            preserved-paths)]
     final-m)))

(defn scrubbed-err [exceptions e]
  ;; TODO: Check if e is an ExceptionInfo and include a scrubbed form of its
  ;; data map.
  (when e
    (let [message (.getMessage e)
          scrubbed-emsg (when message (remove-pii exceptions message))]
      (proxy [Exception] [scrubbed-emsg]
        (getMessage [] (str "[" (-> e .getClass .getName) "] " scrubbed-emsg))
        (getStackTrace [] (.getStackTrace e))
        (getCause [] (when-let [cause (.getCause e)]
                       (scrubbed-err exceptions cause)))))))

(defn ^:deprecated scrub-pii-middleware [excluded-formats exceptions]
  (fn [{:keys [vargs ?msg-fmt ?err] :as r}]
    (if-not (excluded-formats ?msg-fmt)
      (assoc r :vargs (map (fn [varg] (remove-pii exceptions (str varg)))
                           vargs)
             :?err (scrubbed-err exceptions ?err))
      r)))
