(ns perf.core
  (:require [taoensso.timbre :as log]
            [taoensso.tufte :as tufte]))

(defn- format-ns [^:Long time-ns]
  (->> time-ns
       (* 1E-6)
       (format "%.3fms")))

(defn- format-profile-entry [[fieldname {:keys [count mean min max]}]]
  (format "name=%s num-entries=%d mean=%s min=%s max=%s"
          (name fieldname)
          count
          (format-ns mean)
          (format-ns min)
          (format-ns max)))

(defn- format-profile-stats [stats]
  (let [stats-str (some->> stats
                           :stats
                           :id-stats-map
                           (map format-profile-entry)
                           (clojure.string/join ", "))]
    (or stats-str
        (str "Bad Tufte stats supplied! " (pr-str stats)))))

(defn bind-to-timbre!
  "Adds a handler to Tufte to log stats to Timbre using a standard format.
  Takes an atom called `log-perf?` which, when dereferenced, tells us whether
  performance logging is currently enabled."
  [log-perf?]
  (when @log-perf?
    (tufte/add-handler!
      :log-stats
      (fn [stats]
        (log/debug (format-profile-stats stats))))))

(defmacro p [k & body] `(tufte/p ~k ~@body))

(def ^:private names-profiled (atom #{}))

(defmacro defpfn
  "Defines a function whose body needs to be profiled. The same as defining a
  function like this: `(defn x \"A function\" [y] (p :x (stuff y))`"
  [name doc argv & body]
  (let [k (keyword name)]
    (when (@names-profiled k)
      (log/fatalf "Function name already profiled: %s" k))
    (swap! names-profiled conj k)
    `(defn ~name ~doc ~argv (p ~k ~@body))))

(defn wrap-timing
  "Provides a compojure middleware for performing Tufte profiling. Takes an atom
  called `log-perf?` which, when dereferenced, tells us whether performance
  logging is currently enabled. Also takes the standard handler. By default, all
  requests are profiled with the name `request`. Otherwise, takes a function to
  generate the stat name from the request"
  ([log-perf? handler]
   (wrap-timing log-perf? (constantly "request") handler))
  ([log-perf? stat-name-fn handler]
   (fn [request]
     (let [f (partial handler request)
           stat-name (stat-name-fn request)]
       (if @log-perf?
         (tufte/profile {:dynamic? true} (p stat-name (f)))
         (f))))))

(defn stat-name-from-request-uri
  "Grabs the route from a ring request and turns it into a stat name"
  [{[method path] :compojure/route}]
  (format "%s--%s"
          (name method)
          (-> path
              (clojure.string/replace-first "/" "")
              (clojure.string/replace "/" "--")
              (clojure.string/replace ":" ""))))

(comment
  (defpfn best-function-ever
    "A function that does amazing things"
    [x y]
    (+ x y))

  (bind-to-timbre! (delay true))
  (tufte/profile {:dynamic? true} (best-function-ever 5 9))
  )
