(ns perf.core
  (:require [taoensso.timbre :as log]
            [taoensso.tufte :as tufte]))

(defn- format-ns [^:Long time-ns]
  (->> time-ns
       (* 1E-6)
       (format "%.3fms")))

(defn- format-profile-entry [[fieldname {:keys [count mean min max]}]]
  (format "name=%s num-entries=%d mean=%s min=%s max=%s"
          (name fieldname)
          count
          (format-ns mean)
          (format-ns min)
          (format-ns max)))

(defn- format-profile-stats [stats]
  (let [stats-str (some->> stats
                           :stats
                           :id-stats-map
                           (map format-profile-entry)
                           (clojure.string/join ", "))]
    (or stats-str
        (str "Bad Tufte stats supplied! " (pr-str stats)))))

(defn bind-to-timbre!
  "Adds a handler to Tufte to log stats to Timbre using a standard format.
  Takes an atom called `log-perf?` which, when dereferenced, tells us whether
  performance logging is currently enabled."
  [log-perf?]
  (when @log-perf?
    (tufte/add-handler!
      :log-stats
      (fn [stats]
        (log/debug (format-profile-stats stats))))))

(defmacro p [k & body] `(tufte/p ~k ~@body))

(defn wrap-timing
  "Provides a compojure middleware for performing Tufte profiling. Takes an atom
  called `log-perf?` which, when dereferenced, tells us whether performance
  logging is currently enabled. Also takes the standard handler"
  [log-perf? handler]
  (fn [request]
    (let [f (partial handler request)]
      (if @log-perf?
        (tufte/profile {:dynamic? true} (p :request (f)))
        (f)))))
