(ns health-check.core
  (:require [clj-http.client :as http])
  (:import (java.lang.management ManagementFactory)
           (java.text NumberFormat)))

(defn response
  "Indicates a service is either up or down. If an `error` is not `nil`, then
  you will get a map that looks like `{:up? false :error error}`. Otherwise,
  the map will look like `{:up? true :status status}`. It will then
  be merged into the `base-response`."
  [base-response error & [status]]
  (let [up? (boolean (not error))]
    (merge base-response
           {:up? up?}
           (if up? {:status status}
                   {:error (if (instance? Throwable error)
                             (.getMessage error) error)}))))

(defn failure
  "Indicates a service is down. If only an `error` is given, then
  you will get just a map that looks like `{:up? false :error error}`.
  Otherwise, the given `base-response` will be merged into the above map"
  ([error] (failure nil error))
  ([base-response error] (response base-response error)))

(defn success
  "Indicates a service is up. If only a `status` is given, then
  you will get just a map that looks like `{:up? true :status status}`.
  Otherwise, the given `base-response` will be merged into the above map"
  ([status] (success nil status))
  ([base-response status] (response base-response nil status)))

(defn- ping-service [[k f]] {k (if (fn? f) (f) f)})

(defn- service-up? [v]
  (or (not (and (map? v) (contains? v :up?)))
      (:up? v)))

(defn health-check
  "Checks various services and aggregates their statuses up
  into a single overall status with descriptive data

      m -- map of service name and function/value
           which represent the status of those
           services

      af -- an aggregation function which takes all the sub-statuses
            of a structure and converts them into a single boolean value
            defaults to `every?`

  If it is a function it should return the result of a call to either `failure`,
  `success` or `response`. Alternatively, the function could return any value
  which will be passed straight through without contributing to the overall
  status.

  For example:

      (health-check
        {:version \"1.2.3\"
         :up? (fn []
                (try (success (hit-db))
                  (catch Exception e
                    (failure \"db down\")))})"
  ([m]
   (health-check every? m))
  ([af m]
   (let [raw-results (apply merge (doall (pmap ping-service m)))
         up? (af service-up? (vals raw-results))]
     (assoc raw-results :up? up?))))

(defn network-test
  "A generic network test function using clj-http. Give it a URL to hit and
  a function to invoke if an exception occurs. The function will receive the
  exception as its first parameter."
  [test-url error-fn & {:as override-options}]
  (try
    (success (select-keys (http/head test-url (merge
                                               {:throw-exceptions false
                                                :socket-timeout 1500
                                                :connection-timeout 1500}
                                               override-options))
                          [:status :body :request-time :error]))
    (catch Throwable t
      (error-fn t)
      (success (.getMessage t)))))

(defn jvm-settings
  "Produces the arguments that were sent to the currently running process."
  []
  (-> (ManagementFactory/getRuntimeMXBean) .getInputArguments sort))

(def ^:private mb (* 1024 1024))

(defn- memory-value [nf v] (.format nf (/ v mb)))

(defn memory-info
  "Produce a map that contains memory usage information."
  []
  (let [runtime (Runtime/getRuntime)
        nf (NumberFormat/getInstance)
        format-value (comp (partial format "%sMB") (partial memory-value nf))
        free-memory (.freeMemory runtime)
        allocated-memory (.totalMemory runtime)
        max-memory (.maxMemory runtime)]
    (reduce-kv (fn [m k v] (assoc m k (format-value v)))
               {}
               {:free-memory free-memory
                :allocated-memory allocated-memory
                :max-memory max-memory
                :total-free (+ free-memory (- max-memory allocated-memory))})))
