(ns health-check.core)


(defn response
  "Indicates a service is either up or down. If an `error` is not `nil`, then
  you will get a map that looks like `{:up? false :error error}`. Otherwise,
  the map will look like `{:up? true :status status}`. It will then
  be merged into the `base-response`."
  [base-response error & [status]]
  (let [up? (boolean (not error))]
    (merge base-response
           {:up? up?}
           (if up? {:status status}
                   {:error (if (instance? Throwable error)
                             (.getMessage error) error)}))))

(defn failure
  "Indicates a service is down. If only an `error` is given, then
  you will get just a map that looks like `{:up? false :error error}`.
  Otherwise, the given `base-response` will be merged into the above map"
  ([error] (failure nil error))
  ([base-response error] (response base-response error)))

(defn success
  "Indicates a service is up. If only a `status` is given, then
  you will get just a map that looks like `{:up? true :status status}`.
  Otherwise, the given `base-response` will be merged into the above map"
  ([status] (success nil status))
  ([base-response status] (response base-response nil status)))

(defn- ping-service [[k f]] {k (if (fn? f) (f) f)})

(defn- service-up? [v]
  (or (not (and (map? v) (contains? v :up?)))
      (:up? v)))

(defn health-check
  "Checks various services and aggregates their statuses up
  into a single overall status with descriptive data

      m -- map of service name and function/value
           which represent the status of those
           services

      af -- an aggregation function which takes all the sub-statuses
            of a structure and converts them into a single boolean value
            defaults to `every?`

  If it is a function it should return the result of a call to either `failure`,
  `success` or `response`. Alternatively, the function could return any value
  which will be passed straight through without contributing to the overall
  status.

  For example:

      (health-check
        {:version \"1.2.3\"
         :up? (fn []
                (try (success (hit-db))
                  (catch Exception e
                    (failure \"db down\")))})"
  ([m]
   (health-check every? m))
  ([af m]
   (let [raw-results (apply merge (doall (map ping-service m)))
         up? (af service-up? (vals raw-results))]
     (assoc raw-results :up? up?))))
