(ns oc.lib.text
  "Functions related to processing text."
  (:require [clojure.string :as s]
            [cuerdas.core :as str]
            #?(:clj [jsoup.soup :as soup])))

(defn attribution
  "
  Given the number of distinct authors to mention, the number of items, what to call the
  item (needs to pluralize with just an 's'), and a sequence of authors of the items
  to attribute (sequence needs to be distinct'able, and have a `:name` property per author),
  return a text string that attributes the authors to the items.

  E.g.

  (attribution 3 7 'comment' [{:name 'Joe'} {:name 'Joe'} {:name 'Moe'} {:name 'Flo'} {:name 'Flo'} {:name 'Sue'}])
  '7 comments by Joe, Moe, Flo and others'
  "
  [attribution-count item-count item-name authors]
  (let [distinct-authors (distinct authors)
        author-names (map :name (take attribution-count distinct-authors))
        more-authors? (> (count distinct-authors) (count author-names))
        multiple-authors? (> (count author-names) 1)
        author-attribution (cond
                              ;; more distinct authors than we are going to mention
                              more-authors?
                              (str (clojure.string/join ", " author-names) " and others")

                              ;; more than 1 author so last mention needs an "and", not a comma
                              multiple-authors?
                              (str (clojure.string/join ", " (butlast author-names))
                                                        " and "
                                                        (last author-names))

                              ;; just 1 author
                              :else
                              (first author-names))]
    (str item-count " " item-name (when (> item-count 1) "s") " by " author-attribution)))


(defn strip-xss-tags
  "
   Current xss tags are script, style, and input.
  "
  [data]
  (when data (s/replace data #"(?i)<\/?((script|style|input){1})(\s?[^<>]*)>" "")))

(defn- clean-body-text [body]
  (-> body
    (s/replace #"&nbsp;" " ")
    (str/strip-tags)
    (str/strip-newlines)))

(def body-words 20)

(defn truncated-body [body]
  (let [clean-body (if-not (clojure.string/blank? body)
                     (clean-body-text (.text (soup/parse body)))
                     "")
        splitted-body (clojure.string/split clean-body #" ")
        truncated-body (filter not-empty
                        (take body-words ;; 20 words is the average sentence
                         splitted-body))
        reduced-body (str (clojure.string/join " " truncated-body)
                      (when (= (count truncated-body) body-words)
                        "..."))]
    reduced-body))

(defn- thumbnail-elements [body]
  #?(:clj
     (let [parsed-body (soup/parse body)
           els (.select parsed-body "img:not(.emojione), iframe")]
      {:elements els
       :count (count els)})
     :cljs
     (let [$body (js/$ (str "<div>" activity-body "</div>"))
           els (js->clj (js/$ "img:not(.emojione), iframe" $body))]
       {:elements els
        :count (.-length els)})))

(defn- $el [el]
  #?(:clj
      el
     :cljs
      (js/$ el)))

(defn- tag-name [el]
  #?(:clj
      (.tagName el)
     :cljs
      (.-tagName el)))

(defn first-body-thumbnail
  "Given an entry body get the first thumbnail available.
  Thumbnail type: image, video or chart."
  [activity-body]
  (let [{:count els-count :elements thumb-els} (thumbnail-elements activity-body)
        found (atom nil)]
    (dotimes [el-num els-count]
      (let [el (aget thumb-els el-num)
            $el ($el el)]
        (when-not @found
          (if (= (s/lower (tag-name el)) "img")
            (let [width (.attr $el "width")
                  height (.attr $el "height")]
              (when (and (not @found)
                         (or (<= width (* height 2))
                             (<= height (* width 2))))
                (reset! found
                  {:type "image"
                   :thumbnail (if (.attr $el "data-thumbnail")
                                (.attr $el "data-thumbnail")
                                (.attr $el "src"))})))
            (reset! found {:type (.attr $el "data-media-type") :thumbnail (.attr $el "data-thumbnail")})))))
    @found))