(ns datomscript.core.om.parser
  (:require
   [om.tempid]
   [clojure.set :as set]
   [clojure.string :as string]
   [om.next.protocols :as om.protocols]
   [om.next.impl.parser :as om.parser]
   [om.util]   
   [clojure.walk :as walk]
   #?@(:clj [[datomic.api :as d]
             [clojure.tools.logging :as log]
             [om.next.server :as om]]
       :cljs [[datascript.core :as d]                                          
              [om.next :as om]
              [datomscript.client.om.sha256 :as sha256]
              [datomscript.client.datascript :as utils.ds]
              [datomscript.client.om.subscription]
              [datomscript.client.om.syncing]
              [datomscript.client.om.db]
              [om.next.impl.parser :as om.parser]              
              [datomscript.client.om.core :as om.core]]
       )))

;; This can be replaced
;; This should probably be a prewalk
;; However, what is happening is that if you have something that only returns
;; db/id, it is either an ident, or a pull expression that didn't get all of the
;; entity.  If the value of the id has an db/ident value, use that.
;; Otherwise, return the db/id resource/eid and the model/type
(defn replace-pull-idents [data db schema]
  (walk/postwalk
   (fn [d]          
     (if (and (vector? d)
              (= (:db/valueType (get schema (first d))) :db.type/enum)
              (map? (second d))
              (= (count (second d)) 1)
              (= (-> d second ffirst) :db/id))
       [(first d) (:db/ident (d/entity db (:db/id (second d))))]       
       d))
   data))

(defn is-recursive-query? [query]
  (some (fn [q] (or (number? (:query q))
                    (= '... (:query q)))) (:children (om.parser/query->ast query))))

(defn has-wildcard? [query]
  (let [has-wildcard (atom false)]
    (walk/postwalk
     (fn [q]
       (if (= q '*)
         (do
           (reset! has-wildcard true)
           q)
         q))
     query)
    @has-wildcard))

(defn flatten-stuff [args]
    (if (vector? args)
      (reduce
       (fn [acc q]         
         (if (vector? q)
           (vec (concat acc (flatten-stuff q) ))
           (conj acc q)))
       []
       args)
      args))

#?(:cljs
   (defn force-flatten [args]
     (if (vector? args)
       (reduce
        (fn [acc q]         
          (if (vector? q)
            (vec (concat acc (flatten-stuff q) ))
            (conj acc q)))
        []
        args)
       args)))

#?(:cljs
   (defn process-roots [send-map]
     (-> (om/process-roots send-map)
         :query
         force-flatten)))

#?(:cljs
   (defn db-ids->om-ids [data]
     (clojure.walk/prewalk
      (fn [d]
        (if-let [id (:db/id d)]
          (assoc d :db/id (om.tempid/tempid id))
          d))
      data)))

#?(:cljs
   (defn om-ids->db-ids [data]
     (clojure.walk/prewalk
      (fn [d]
        (if-let [[_ id] (find d :db/id)]
          (assoc d :db/id 
                 (if (om.tempid/tempid? d)
                   (.-id id)                                
                   (utils.ds/tempid)))
          d))
      data)))

#?(:cljs
   (defn db-ids->tmpids [response]
     (clojure.walk/prewalk
      (fn [r]
        (if (:db/id r)
          (assoc r :db/id (datomscript.client.om.db/tempid))
          r))
      response)))

#?(:cljs
   (defn starts-with-app? [str]
     (and str (boolean (seq (re-find #"^app" str))) )))

#?(:cljs
   (defn is-app-key? [key]
     (when (and key (or (keyword? key) (symbol? key)))    
       (let [key-to-check (if (namespace key)
                            (namespace key)
                            (name key))]
         (starts-with-app? key-to-check)))))

#?(:cljs
   (defn remove-app-keys-from-payload [payload]
     (walk/prewalk
      (fn [d]
        (if (map? d)
          (reduce-kv (fn [acc k v] (if (is-app-key? k)
                                     acc
                                     (assoc acc k v))) {} d)
          d))
      payload)))

#?(:cljs (declare remove-app-keys-from-ast))

#?(:cljs
   (defn remove-app-keys-from-children [ast]
     (if-let [children (:children ast)]
       (let [new-children (reduce
                           (fn [acc c]
                             (if-let [new-c (remove-app-keys-from-ast c)]
                               (conj acc new-c)
                               acc))
                           []
                           children)]
         (if (seq new-children)
           (assoc ast :children new-children :query (om/ast->query new-children))
           (dissoc ast :children :query)))    
       ast)))

#?(:cljs
   (defn remove-app-keys-from-ast-params [ast]
     (if-let [params (:params ast)]
       (assoc ast :params (remove-app-keys-from-payload params))
       ast)))

#?(:cljs
   (defn remove-app-keys-from-ast [ast]  
     (if (is-app-key? (:dispatch-key ast))
       nil
       (-> ast 
           remove-app-keys-from-children
           remove-app-keys-from-ast-params))))

#?(:cljs
   (defn remove-app-keys-from-query [query]
     (om/ast->query (remove-app-keys-from-ast (om/query->ast query)))))

#?(:cljs
   (defn checksum-expr [expr]
     (datomscript.client.om.sha256/sha256 (pr-str expr))))

#?(:cljs
   (defn checksum-ident [expr]
     (let [{:keys [dispatch-key] :as ast} (om.parser/expr->ast expr) ]
       [dispatch-key (checksum-expr expr)])))

#?(:cljs
   (defn add-checksum-ident-to-expr [expr]
     (let [{:keys [dispatch-key] :as ast} (remove-app-keys-from-ast (om.parser/expr->ast expr)) ]
       (om.parser/ast->expr
        (assoc ast :key [dispatch-key (checksum-expr expr)])))))

#?(:cljs (defn add-checksum-ident-to-query [query]
           (mapv add-checksum-ident-to-expr query)))

#?(:cljs
   (defn cache-ast-key [ast & [params]]
     (if-let [i (get-in ast [:params :app/params :cache-ident])]
       i
       (checksum-expr
        (om.parser/ast->expr
         (remove-app-keys-from-ast ast))))))

#?(:cljs
   (defn checksum-ident-expr [expr]
     [:app/cache (cache-ast-key (om.parser/expr->ast expr)
                                (:params (om.parser/expr->ast expr)))]))

#?(:cljs
   (defn cache-ast [ast & [params]]
     (assoc ast :key [(:dispatch-key ast) (cache-ast-key ast params)])))

#?(:cljs
   (defn extract-app-params [query]
     (get-in (om.parser/expr->ast query)
             [:params
              :app/params])))

#?(:cljs
   (defn uniform-response-from-query [query value]
     (reduce
      (fn [acc req]
        (assoc acc (:key (om.parser/expr->ast req)) value))
      {}
      query)))

#?(:cljs
   (defn om-ids->resource-eids [response]
     (reduce-kv
      (fn [acc om-id resource-eid]     
        (conj acc
              {:db/id (.-id om-id)
               :resource/eid resource-eid}))
      []
      response)))

#?(:cljs
   (defn resolve-tmp-ids [data resolve]
     (walk/postwalk
      (fn [d]
        (if (neg? d)
          (resolve d)
          d))
      data)))

#?(:cljs
   (defn set-component-refs! [{:keys [component refs resolve-fn]}]
     (om.core/set-component-refs! component (resolve-tmp-ids refs resolve-fn))))

#?(:cljs
   (defn set-component-query-params! [{:keys [component query params resolve-fn]}]     
     (om.core/set-query-params! component {:query query
                                            :params (resolve-tmp-ids
                                                     params resolve-fn) } )))

#?(:cljs
   (defn transact-action [{:keys [conn component ast]} key params]  
     (when-let [app-tx-data (get-in params [:app/params :tx-data])]    
       (try
         (let [tx-report (utils.ds/transact! conn app-tx-data)
               resolve-fn (partial d/resolve-tempid
                                   (:db-after tx-report)
                                   (:tempids tx-report))
               _ (when-let [refs (get-in params [:app/params :component :refs])]
                   (set-component-refs! {:component component
                                         :refs refs
                                         :resolve-fn resolve-fn}))
               _ (let [{:keys [query params]} (get-in params [:app/params :component])]                
                   (when (or query params)
                     (set-component-query-params! {:component component
                                                   :query query
                                                   :params params
                                                   :resolve-fn resolve-fn})))]
           (when-let [tx-report-atom (get-in params [:app/params :tx-report])]
             (reset! tx-report-atom tx-report)))      
         (catch js/Error e
           (.error js/console e))))))

#?(:cljs
   (defn specific-transact [{:keys [conn ast] :as env} key params]
     {:remote (when (get-in params [:remote])             
                (assoc-in ast [:params] (merge (:remote params)
                                          {:app/params (:app/params params)})))
      :action (fn [_]
                (transact-action env key params))}))

#?(:cljs
   (defn model-transact [{:keys [conn ast] :as env} key params]  
     {:remote (when (get-in params [:remote])
                (assoc-in ast [:params] (assoc (dissoc params :remote)
                                               :tx-data (:remote params))))
      :action (fn [_]
                (transact-action env key params))}))

#?(:cljs
   (defn read-app-attr [{:keys [conn parser entity query]}]     
     {:value (:app/attr entity)}))

#?(:cljs
   (defn ast-sub-wrap [ast params]  
     (assoc (om.parser/expr->ast
             `(subscription/update {:new [~(om.parser/ast->expr (cache-ast ast))]}))
            :query-root true)))

#?(:cljs
   (defn parse-by-app-db-id [{:keys [parser ast target conn query] :as env} key params]
     {:value (when-let [id (-> ast :key second)]
               (parser (assoc (dissoc env :query)
                              :entity (d/entity @conn id))
                       query))}))

#?(:cljs
   (defn by-ident-entity
     [{:keys [parser ast target conn query entity] :as env} key params]     
     (let [[key-attr id] (:key ast)
           key (if (= key-attr :db/id)
                 id
                 key)]
       (and id (d/entity @conn key )))))

#?(:cljs
   (defn custom-ast-remote [{:keys [ast query]} key params]     
     (if (= (get-in params [:app/params :remote]) :sub)
       (ast-sub-wrap ast params)
       ;; Should be wrapping the ast in a cache-ast
       (assoc (cache-ast ast) :query-root true))))

#?(:cljs
   (defn defer-ast-remote [{:keys [parser target query] :as env} key params]     
     (let [new-env (dissoc env :query)        
           q (parser new-env query target)]    
       (when (seq q)
         (assoc (om/query->ast q ) :query-root true)))))

#?(:cljs
   (defn cache-value?
     [ast db]     
     (let [cache-key (cache-ast-key ast)
           ent (d/entity db [:app/cache cache-key])]
       (not (nil? (:app/value ent))))))

#?(:cljs
   (defn by-ident-remote
     [{:keys [parser ast target conn query entity] :as env} key params]     
     (let [ent (by-ident-entity env key params)           
           new-env (assoc env :entity ent)           
           eid (or (when-let [k (get-in params [:app/params :ident-key])]
                     (get ent k))
                 (:resource/eid ent))           
           target-value
           ,(when (and eid (get-in params [:app/params :remote]))
              (let [ast' (om.parser/expr->ast
                          `({:db/pull ~query}
                            ~(merge
                              params
                              {:ident [:resource/eid eid] })))]                
                (custom-ast-remote (assoc new-env :ast ast' ) key params)))]    
       (when (seq target-value)
         {target target-value}))))

#?(:cljs
   (defn by-ident-value
     [{:keys [parser ast target conn query entity] :as env} key params]     
     (let [key (:key ast)]    
       (if (vector? key)
         (do           
           (when-let [ent (by-ident-entity env key params)]             
             (let [new-env (assoc (dissoc env :query) :entity ent)]                          
               {:value (if (and (= (first key) :db/id)
                             (or (is-recursive-query? query) (has-wildcard? query)))
                         (d/pull @conn query (:db/id ent))
                         (parser new-env query))})))
         {:value (get entity key)}))))

#?(:cljs
   (defn by-ident
     [{:keys [parser ast target conn query entity] :as env} key params]     
     (let [[ast key] (if (and (vector? (:key ast ))
                           (keyword? (second (:key ast))))
                       (let [[k id] (:key ast)
                             db-id (:db/id (d/entity @conn [:app/id id]))
                             new-key [k db-id]]
                         [(assoc ast :key new-key)
                          new-key])
                       [ast key])
           env (assoc env :ast ast )]       
       (if target
         (when (vector? (:key ast))
           (by-ident-remote env key params))
         (by-ident-value env key params)))))

#?(:cljs
   (defn by-cache-value
     [{:keys [parser ast target conn query entity] :as env} key params]
     (let [cache-key (cache-ast-key ast params)
           ent (d/entity @conn [:app/cache cache-key])]       
       {:value     
        (when ent
          (set/rename-keys
           (d/pull @conn [:app/value :app/syncing? :app/sub-value] (:db/id ent))
           {:app/value :value
            :app/sub-value :sub-value
            :app/syncing? :syncing?}))})))

#?(:cljs
   (defn by-cache
     [{:keys [parser ast target conn query entity] :as env} key params]     
     (if (and target
              (get-in params [:app/params :remote])
              (or
               (not (get-in params [:app/params :cache?]))
               (and (get-in params [:app/params :cache?])
                    (not (cache-value? ast @conn) ))))
       {target (custom-ast-remote env key params)}
       (by-cache-value env key params))))

#?(:cljs
   (defn by-widget
     [{:keys [parser ast target conn query] :as env} key params]     
     (if target
       (let [q (if (map? query)
                 (reduce-kv
                  (fn [acc k v]
                    (vec
                     (concat
                      acc 
                      (parser (dissoc env :query) [{k v}] target))))
                  [] query)
                 (parser (dissoc env :query) query target))]         
         (when (seq q)
           {target (assoc (om/query->ast q ) :query-root true)}))       
       {:value (if (map? query)
                 (reduce-kv
                  (fn [acc k v]
                    (merge
                     acc 
                     (parser (dissoc env :query)
                             [{k v}])))
                  {} query)
                 (parser (dissoc env :query) query))})))

(defn entity-by-query [entity {:keys [parser query] :as env}]
  (parser (assoc (dissoc env :query) :entity entity) query))

(defn cardinality-many-ref-attr [{:keys [parser remote? entity query] :as env} key _]
  {:value (mapv
           (fn [ref]
             (entity-by-query ref env))
           (get entity key))})

(defn cardinality-one-ref-attr [{:keys [parser target ast entity query] :as env}  key _]  
  {:value (entity-by-query (get entity key) env)})

(defn value-attr [{:keys [entity target]} key params]  
  {:value (get entity key)})

(defn is-schema-value-key? [schema-map]
  "Is this schema attribute a valueType of something other than ref."
  (not= (:db/valueType schema-map) :db.type/ref))

(defn is-ident-ref? [schema-map]
  "Is this a schema attribute that can be found by an ident and is a unique-identity"  
  (= (:db/unique schema-map) :db.unique/identity))

(defn is-cardinality-many? [schema-map]
  (= (:db/cardinality schema-map) :db.cardinality/many))

(defn is-single-ref? [schema-map]
  "Is this a schema attribute a single entity reference"
  (and (= (:db/valueType schema-map) :db.type/ref)
       (not (is-cardinality-many? schema-map))))

(defn is-single-back-ref? [schema-map]
  "Determine if the schema is a back reference returning only a single
  map value, which occurs when the schema attribute is a component."
  (and (= (:db/valueType schema-map) :db.type/ref)
       (:db/isComponent schema-map)))

(defn is-multi-ref? [schema-map]
  "Determine if the schema is a cardinality many reference type"
  (and (= (:db/valueType schema-map) :db.type/ref)
       (is-cardinality-many? schema-map)))

(defn is-multi-back-ref? [schema-map]
  "Determine if the schema is a mutli back reference, meaning
  it will return a vector of entries."
  (and (= (:db/valueType schema-map) :db.type/ref)
       (not (:db/isComponent schema-map))))

(defn get-back-reference-key [k]
  (when (and (namespace k) (re-find #"_" (name k)) )
    (keyword (namespace k) (string/replace (name k) #"_" ""))))

(defn is-back-ref? [key]
  (boolean (re-find #"^[_].*?" (name key)))) 

(defn classify-datomic-attr [datomic-schema-attr]  
  (cond
    #?(:cljs (is-ident-ref? datomic-schema-attr)) #?(:cljs by-ident)
    (is-single-ref? datomic-schema-attr) cardinality-one-ref-attr
    (is-multi-ref? datomic-schema-attr) cardinality-many-ref-attr    
    :else value-attr))

(defn classify-datomic-back-attr [datomic-schema-attr]
  (cond
    (is-single-back-ref? datomic-schema-attr) cardinality-one-ref-attr
    (is-multi-back-ref? datomic-schema-attr) cardinality-many-ref-attr
    :else nil))

(defn get-datomic-key-fn [schema key]
  (if (is-back-ref? key)
    (when-let [back-ref-attr (some->> key
                               get-back-reference-key
                               (get schema))]
      (classify-datomic-back-attr back-ref-attr))
    (classify-datomic-attr (get schema key))))
#?(:cljs
   (defn page-loading? [{:keys [conn]} _ _]
     {:value (boolean (d/q '[:find ?x .
                             :in $
                             :where                             
                             [?x :app/syncing? true]]
                        @conn))}))


#?(:cljs
   (defn env-state->conn [env]
     "Given an enviornment map that goes to the parser functions, it adds a key
  called `conn`, which is an alias for state."
     (assoc env :conn (:state env))))

#?(:cljs
   (defn -run-route-fn [f env key params]
     "Helper function for tying a route to a dispatch key.  Prints an error message
  to the console when there isn't one."
     (if f
       (f (env-state->conn env) key params)
       (do
         (.error js/console "There is no route for the key" (pr-str key))
         nil))))

#?(:cljs
   (defn -get-widget [key]
     (when (->> (str key)
                (re-find #"[-][>]"))
       by-widget))) 

#?(:cljs
   (defn by-ident-join
     [{:keys [parser target ast] :as env} key params]
     (let [new-query {key (second (:key ast))}
           new-ast (om.parser/expr->ast new-query)
           p-fn (partial parser (assoc env :ast new-ast) [new-query])]    
       (if target
         {target (p-fn target)}
         {:value (om.util/join-value (p-fn))}))))

#?(:cljs
   (defn read-fn [{:keys [schema config]}]  
     (fn [{:keys [ast state query] :as env} key params]       
       (let [read-f (if (vector? key)
                      by-ident-join
                      (or (get config key)
                          (-get-widget key)
                          (get-datomic-key-fn (utils.ds/extract-schema state) key)))]      
         (-run-route-fn read-f env key params)))))

#?(:cljs
   (defn mutate-fn [{:keys [schema config transact!]}]
     (fn [env key params]      
       (-run-route-fn (get config key ) (assoc env :transact! transact!) key params))))

#?(:clj
   (defn om-ids->tmp-ids [data]
     (let [id-map (atom {})
           data-with-tmp-ids (walk/postwalk
                              (fn [d]        
                                (if (om.tempid/tempid? d)
                                  (let [tmp-id (d/tempid :db.part/user)]
                                    (swap! id-map assoc tmp-id d)
                                    tmp-id)
                                  d))
                              data)]
       [@id-map data-with-tmp-ids])))

#?(:clj
   (defn restore-ids [data tmp-id->om-id]  
     (walk/postwalk
      (fn [d]
        (if (= (type d) datomic.db.DbId)
          (get tmp-id->om-id d)
          d))
      data)))

#?(:clj
   (defn force-eids [data db]
     (walk/prewalk
      (fn [d]
        (if (and (map? d) (:db/id d) (not (:resource/eid d)))
          (assoc d :resource/eid (:resource/eid (d/entity db (:db/id d))))
          d))
      data)))

#?(:clj
   (defn -extract-error [e]
     (if (or (= (type e) datomic.impl.Exceptions$IllegalArgumentExceptionInfo)
             (= (type e) datomic.impl.Exceptions$IllegalStateExceptionInfo))
       {:message (.getMessage e) :details ""}  
       (if-let [err (ex-data e)]
         (if (:message err)
           {:message (:message err) :details (:details err)}
           {:message "There was an unknown server error"})
         {:message (.getMessage e)
          :details (str (type e))}))))

#?(:clj
   (defn merge-results [data]
     "Merges the :result portion of an om response back into the main response, 
  rather than having them separated.
  e.g. {:x 5 :result {:y 7}} becomes {:x 5 :y 7}"
     (walk/postwalk
      (fn [d]     
        (if (:result d)
          (dissoc (merge d (:result d)) :result)
          d))
      data)))

#?(:clj
   (defn convert-errors [resp]
     "Extracts any exception error messages and places them into the om.next/error 
  portion.  Also returns metadata with ^was-error? set to true."
     (let [was-error? (atom false)]
       (with-meta
         (walk/prewalk
          (fn [d]
            (if (and (map? d) (:om.next/error d))
              (do
                (reset! was-error? true)
                (assoc d :om.next/error (-extract-error (:om.next/error d))))
              d))
          resp)
         {:was-error? @was-error?}))))

#?(:clj
   (defn run-route-fn [f env key params]
     (if f
       (f env key params)
       (throw (ex-info "There is no route"
                       {:message "There is no route"
                        :details {:params params
                                  :key key}})))))

#?(:clj
   (defn read-fn [read-config]  
     (fn [{:keys [query entity] :as env} key params]       
       (run-route-fn (get read-config key )
                     env
                     key
                     params))))

#?(:clj
   (defn mutate-fn [mutate-config]  
     (fn [env key params]       
       (run-route-fn (get mutate-config key )
                     env
                     key
                     params))))

#?(:clj
   (defn make-parser [{:keys [read-config mutate-config]} ]     
     (om/parser {:read (read-fn read-config)
                 :mutate (mutate-fn mutate-config)})))

#?(:clj
   (defn manage-db-id-map [data db]
     (clojure.walk/prewalk
       (fn [d]
         (if (and (map? d) (= (count d) 1) (:db/id d))
           (if-let [ident (:db/ident (d/entity db (:db/id d)))]
             ident
             (assoc d :resource/eid (:resource/eid (d/entity db (:db/id d)))))
           d))
       data)))

#?(:clj
   (defn pipeline [{:keys [config services user-token query] :as args}]     
     (let [[tmp-id->om-id new-query] (om-ids->tmp-ids query)]       
       (-> ((make-parser config) (dissoc args :query)
            new-query)
         merge-results
         (restore-ids tmp-id->om-id)           
         convert-errors))))

#?(:clj
   (defn as-of-db-by-eid [db eid]
     (d/as-of db (:db/id (d/entity db [:resource/eid eid])))))

;; Should also do it by a timestamp

#?(:clj
   (defn pull-db [db db-params]     
     (if (and (:db db-params)
              (= (first (:db db-params)) :as-of)
              (instance? java.util.UUID (second (:db db-params))))
       (as-of-db-by-eid db (second (:db db-params)))
       db)))

#?(:clj
   (defn parse-db-params [db params]
     (reduce
      (fn [curr-db p]        
        (cond
          (and (vector? p) (= (first p) :as-of))
          (as-of-db-by-eid db (second p))          
          (= :history p) (d/history curr-db)
          :else curr-db))
      db
      params)))

#?(:clj
   (defn req-q-params-convert [db params]
     (if params
       (map
        (fn [p]          
          (cond 
            (= p :db) db
            (and (vector? p) (= (first p) :db))
            ,(parse-db-params db (rest p))            
            :else p))
        params)
       [db])))

#?(:clj
   (defn extract-req-q [db params]
     (merge 
      {:q (:q params)
       :params (req-q-params-convert db (:params params))}
      (select-keys params [:take :sort-nth :sort-dir]))))

#?(:clj
   (defn sub-q-params-convert [{:keys [deltas db-after db-before]} params]
     (map
      (fn [p]
        (cond 
          (= p :db-after) db-after
          (= p :db-before) db-before
          (= p  :deltas) deltas
          (and (vector? p) (= (first p) :db-after))
          ,(parse-db-params db-after (rest p))
          (and (vector? p) (= (first p) :db-before))
          ,(parse-db-params db-before (rest p))          
          :else p))
      params)))

#?(:clj
   (defn extract-sub-q [tx-report params]
     (let [sub-q-params (:sub-q params)]
       (merge 
        {:q (:q sub-q-params)
         :params (sub-q-params-convert tx-report (:params sub-q-params))}
        (select-keys sub-q-params [:take :sort-nth :sort-dir])))))

(defn sort-dir [d q]
  (if (= d :desc)
    (reverse q)
    q))

#?(:clj
   (defn run-q [{:keys [q params sort-nth sort-dir] take' :take drop' :drop}]
     (let [q-result (cond->> (apply d/q q params)
                      sort-nth (sort-by #(nth % sort-nth))
                      sort-dir (sort-dir (:sort-dir params))
                      drop' (drop drop')
                      take' (take take')) ]
       (if (= (type q-result) java.util.HashSet)
         (into [] q-result)
         q-result))
     ))

(comment  
  
  )




