(ns datomscript.core.model
  (:require #?@(:clj [[datomic.api :as d]
                      [clojure.tools.logging :as log]
                      [clojure.spec :as s]])
            #?@(:cljs [[datascript.core :as d]
                       [cljs.spec :as s]])
            [clojure.string :as string]
            [datomscript.core.spec]
            [clojure.walk :as walk]))

(def vconcat (comp vec concat))

(defn install-attrs [attr-map]
  (reduce-kv
    (fn [acc k v]
      (conj acc (merge {:db.install/_attribute :db.part/db
                        :db/id (d/tempid :db.part/db)
                        :db/ident k}
                  v) ))
    []
    attr-map))

(defn install-enums [enums]
  (mapv #(hash-map :db/ident % :db/id (d/tempid :db.part/user)) enums))

(defn install-schema [attr-map & [enums]]
  (cond-> (install-attrs attr-map)
    enums (vconcat (install-enums enums))))

(defn get-model [models model-type]  
  (get models (keyword (namespace model-type))))

(defn nameify [model]
  (keyword (name (:name model)) "model"))

(defn merge-opts [schema opts]
  (merge schema
         (let [opts' (into #{} opts) ]
           (cond-> {}
             (:fulltext opts') (assoc :db/fulltext true)
             (:component opts') (assoc :db/isComponent true)
             (:nohistory opts') (assoc :db/noHistory true)))))

(defn merge-enums [schema enums]
  (let [ns (name (:ns enums))
        part (:part enums)]    
    (reduce
     (fn [e v]
       (conj e
             {:db/id (d/tempid (or part :db.part/user))
              :db/ident (if (namespace v)
                          v
                          (keyword ns (name v)))}))
     schema
     (:values enums))))

(defn pull-entity-model [entity model]
  (reduce-kv
   (fn [pull k field-map]
     (if (= (:type field-map) :ref)
       {:model/type (:model/type entity)}
       (get entity k)))
   {}
   (:fields model)))

(defn validate-ref-field [field test-value]  
  (if (= (:ref field) :any)
    true
    (let [ref-models (map #(keyword (name %) "model") (if (keyword? (:ref field))
                                                        [(:ref field)]
                                                        (:ref field)))]      
      (or     
       (boolean (get (into #{} ref-models) (:model/type test-value)))
       (str test-value " was not a valid ref model " (pr-str (:ref field)))))))

(defn validate-enum-field [field test-value]
  (let [n (get-in field [:enum :ns])
        enums (into #{} (map (fn [e] (if (namespace e)
                                       e
                                       (keyword (name n) (name e))))
                             (get-in field [:enum :values])))]
    (if (get enums test-value)
      true
      (str test-value " was not a valid enum " (pr-str enums)))))

(defn validate-field [field test-value]  
  (if test-value
    (case (:type field)
      :ref (validate-ref-field field test-value)
      :enum (validate-enum-field field test-value)
      (if-let [validation-fn (:validation field)]
        (if (validation-fn test-value)
          true
          (str test-value (or (:invalid-msg field) " is invalid")))
        true))
    (if (:optional? field)
      true
      "Required Field")))

(defn validate-cardinality-many [field values]
  (let [results (map #(validate-field field %) values)]
    (if (every? true? results)
      true
      (string/join " " (remove true? results)))))

(defn delta-key [{:keys [old new]}]
  (cond
    (and (nil? old) new) [(:model/type new) :added]
    (and old new) [(:model/type new) :updated]
    (and old (nil? new)) [(:model/type old) :removed]))

(defn delta-handler-fns [{:keys [delta handlers]}]
  (let [[model-type status :as delta-key'] (delta-key delta)]
    (->> [(get handlers delta-key')
          (get handlers [model-type :*])
          (get handlers [:* status])
          (get handlers [:* :*])]
         (filter seq)
         (mapcat identity))))

(defn concat-deltas [deltas]
  (mapcat
    (fn [delta]
      (let [[model-type status :as delta-key'] (delta-key delta)]
        (map (fn [d delta-key]
               {:delta d
                :delta-key delta-key})
          (repeat delta)
          [[model-type :*]
           [model-type status]
           [:* status]
           [:* :*]])))
    deltas))

(defn deltas-reducing [{:keys [handler-args deltas handlers]}]  
  (reduce
    (fn [txes {:keys [delta delta-key]}]      
      (vconcat txes (mapcat (fn [f] (let [v (f handler-args delta)]
                                      (if (map? v)
                                        [v]
                                        v)))
                      (get handlers delta-key))))
   []   
   (concat-deltas deltas)))

(defn permission-validation [{:keys [permission-handlers tx-report deltas]}]  
  (let [errs (deltas-reducing {:deltas deltas
                               :handler-args tx-report
                               :handlers permission-handlers})]
    (when (seq errs)
      (throw (ex-info (pr-str "There was a permission error" errs)
                      {:message "There was a permission error"
                       :details errs})))))

(defn append-post-tx-data
  [{:keys [post-tx-handlers tx-report deltas req-tx-data db-id->tmp-id]}]  
  (vconcat
    req-tx-data
    (deltas-reducing {:deltas deltas
                      :handler-args {:tx-report tx-report
                                     :db-id->tmp-id db-id->tmp-id}
                      :handlers post-tx-handlers})))

#?(:clj
   (defn schemafy-datomic [model]
     (let [name' (name (:name model))]       
       (vconcat
         (reduce-kv
           (fn [s field-name {:keys [type enum cardinality unique doc opts] :as field-map}]            
             (vec
               (concat
                 s
                 (cond-> [(cond-> {:db.install/_attribute :db.part/db
                                   :db/id (d/tempid :db.part/db)
                                   :db/ident (keyword name' (name field-name))
                                   :db/cardinality (if (= cardinality :many)
                                                     :db.cardinality/many
                                                     :db.cardinality/one)
                                   :db/valueType (keyword "db.type" (if (= type :enum)
                                                                      "ref"
                                                                      (name type)))}
                            unique (assoc :db/unique (keyword "db.unique" (name unique)))
                            doc (assoc :db/doc doc)
                            opts (merge-opts opts))]              
                   (= type :enum) (merge-enums enum )))))
           []
           (into {} (remove (fn [[k v]] (namespace k)) (:fields model))))
         [{:db/id (d/tempid :db.part/user)
           :db/ident (keyword name' "model")}]))))

#?(:clj
   (def type-schema
     {:db.install/_attribute :db.part/db
      :db/id (d/tempid :db.part/db)
      :db/ident :model/type
      :db/cardinality :db.cardinality/one
      :db/valueType :db.type/ref
      :db/doc "The model type definition"}))

(defn schemafy-as-map [model]
  (let [name' (name (:name model))]       
    (reduce-kv
      (fn [s field-name {:keys [type cardinality unique] :as field-map}]
        (assoc
          s
          (keyword name' (name field-name))
          (cond-> {:db/cardinality (if (= cardinality :many)
                                     :db.cardinality/many
                                     :db.cardinality/one)
                   :db/valueType (keyword "db.type" (name type))}
            unique (assoc :db/unique (keyword "db.unique" (name unique))))
          ))
      {}
      (into {} (remove (fn [[k v]] (namespace k)) (:fields model)))
      )))

#?(:clj
   (defn entity->delta
     [{:keys [db-after db-before tx-data]} {:keys [eid status]}]     
     (let [model-type (:model/type
                       (if (= status :removed)
                         (d/entity db-before eid)
                         (d/entity db-after eid)))]
       (reduce
         (fn [entity [_ a v tx added?]]
           (let [attr-ent (d/entity db-after a)
                 attr-name (:db/ident attr-ent )                
                 path [(if added? :new :old) attr-name]
                 val (if (= (:db/valueType attr-ent) :db.type/ref)
                       (if-let [val-enum (:db/ident (d/entity (if added?
                                                                db-after
                                                                db-before) v))]
                         val-enum
                         v)
                       v)
                 cardinality (:db/cardinality attr-ent)]
             (if (= cardinality :db.cardinality/many)
               (update-in entity path (fnil conj []) val)
               (assoc-in entity path val))))
         (cond-> {:old nil
                  :new nil}
           (= status :added) (assoc-in [:new :db/id] eid)
           (= status :updated) (merge {:new {:db/id eid}
                                       :old {:db/id eid}})
           (= status :removed) (assoc-in [:old :db/id] eid))
         (filter (fn [[e]] (= eid e)) tx-data)))))

#?(:clj
   (defn add-deltas
     [{{:keys [db-after db-before] :as tx-report} :tx-report
       entity-classification :entity-classification}]
     (map (fn [[e status]]
            (let [{:keys [old new] :as deltas}
                  ,(entity->delta
                     tx-report {:status status
                                :eid e})
                  entity (d/entity (if (= status :removed)
                                     db-before
                                     db-after)
                           e)]
              (cond
                ;; Have to potentially deal with a model update change
                ;; where the model must be checked as if it was added
                (and (= status :updated)
                  (not= (:model/type old))
                  (not= (:model/type new)))
                ,{:old nil
                  :new (d/touch entity)}
                ;; Make sure the model types are included
                (= status :updated)
                ,{:old (assoc old :model/type (:model/type entity))
                  :new (assoc new :model/type (:model/type entity))}
                :else deltas)))
       entity-classification)))

#?(:clj
   (defn classify-entities [{:keys [db-before db-after] :as tx-report} tx-data-e]     
     (d/q '[:find ?e ?entity-type
            :in $dbb $dba % [?e ...]
            :where
            [(ground :added) ?entity-type]
            ($dbb entity-exists? ?e ?in-before?)
            ($dba entity-exists? ?e ?in-after?)
            (classify-type ?in-before? ?in-after? ?entity-type)]
       db-before
       db-after
       '[[(entity-exists? ?e ?exists?)
          [?e]                    
          [(ground true) ?exists?]]
         [(entity-exists? ?e ?exists?)
          (not [?e])                    
          [(ground false) ?exists?]]
         [(classify-type ?in-before? ?in-after? ?type)
          [(= ?in-before? false)]
          [(= ?in-after? true)]          
          [(ground :added) ?type]]
         [(classify-type ?in-before? ?in-after? ?type)
          [(= ?in-before? true)]
          [(= ?in-after? true)]          
          [(ground :updated) ?type]]
         [(classify-type ?in-before? ?in-after? ?type)
          [(= ?in-before? true)]
          [(= ?in-after? false)]          
          [(ground :removed) ?type]]]
       tx-data-e)))

#?(:clj
   (s/fdef classify-entities
     :args (s/cat :tx-report :datomscript/tx-report
             :tx-data-e (s/coll-of datomscript.core.spec/long? #{}))
     :ret :datomscript/entity-classification))

#?(:clj 
   (defn add-entity-classification
     [{:keys [tx-data :as tx-report]}]
     (classify-entities tx-report (into #{} (map (fn [[e]] e) tx-data)) )))

#?(:clj
   (s/fdef add-entity-classification
     :args (s/cat :args :datomscript/tx-report )
     :ret :datomscript/entity-classification))

#?(:clj
   (s/fdef classify-entities
     :args (s/cat :tx-report :datomscript/tx-report)
     :ret :datomscript/entity-classification))

(defn spec-field [field]
  "Specs for references need to be compared to the model/type, not the actual
  spec validation."
  (let [is-model? #(and (keyword? %) (= (name %) "model"))
        f (s/form field)
        in? (fn [xs model] (boolean (get xs (:model/type model))))
        every-in? (fn [xs models] (every? (partial in? xs) models))
        model-types #(take-nth 2 (drop 2 %))
        field-name (get (s/registry) field)]    
    (if (is-model? field-name)
      #(or (= (:model/type %) field-name) (= % :any/model))
      (if (sequential? f)        
        (let [ff (first f)]
          (cond
            (and (= ff 'clojure.spec/or) (is-model? (nth f 2)))            
            ,#(boolean (get (into #{} (model-types f)) (:model/type %) ))
            (and (= ff 'clojure.spec/*) (is-model? (second f)))
            ,#(every-in? (into #{} (rest f)) %)
            (and
              (= ff 'clojure.spec/*)
              (list? (second f))
              (= (first (second f)) 'clojure.spec/alt)
              (is-model? (nth (second f) 2)))                        
            ,#(every-in? (into #{} (model-types (second f))) %)            
            :else field
            ))
        field))))

(defn validate-spec-field [attr value]  
  (if (= attr :resource/eid)
    "This is a restricted key"
    (if-let [data (s/explain-data (spec-field attr) value)]
      (::s/problems data)
      true)))

(defn get-model-fields [spec-model]
  (->> (apply hash-map (drop 1 (s/form spec-model)) )
    (reduce-kv
      (fn [fields k v]
        (assoc fields k (into #{} v)))
      {})))

(defn validate-spec-model [spec-model test-value]
  (let [{:keys [req opt]} (get-model-fields spec-model)
        test-value-keys (into #{} (keys test-value))
        spec-keys (into #{} (concat req opt))]    
    (merge
      (into {}
        (zipmap (remove spec-keys test-value-keys)
          (repeat "Is not a valid key")))
      (into {}
        (zipmap (remove spec-keys req)
          (repeat "This is a required key")))            
      (reduce
        (fn [errs field-name]          
          (if (nil? (get test-value field-name))
            errs
            (let [tv (get test-value field-name)                  
                  v (validate-spec-field field-name tv)]
              (if (true? v)
                errs
                (assoc errs field-name v)))))
        {}
        (concat req opt)))))

#?(:clj
   (defn add-delta-model-ref-types [{:keys [delta-model db]}]
     "Reference entities just need their model/type."
     (reduce-kv
       (fn [mv k v]         
         (assoc mv k           
           (if (= (:db/valueType (d/entity db k)) :db.type/ref)
             (if (and (not (vector? v))
                   (:db/ident (d/entity db v)))
               v
               (if (= (:db/cardinality (d/entity db k)) :db.cardinality/many)
                 (mapv (fn [id] {:model/type (:model/type (d/entity db id))}) v)
                 {:model/type (:model/type (d/entity db v))}))
             v)))
       {}
       delta-model)))

#?(:clj
   (defn schema-validation-added [{:keys [delta-model db]} ]     
     (if-let [model-type (:model/type delta-model)]       
       (let [test-value (add-delta-model-ref-types
                          {:db db
                           :delta-model (dissoc delta-model :model/type :db/id)})             
             errors (validate-spec-model
                      model-type
                      test-value)]         
         (when (seq errors)             
           (assoc errors :db/id (:db/id delta-model))))
       {:db/id (:db/id delta-model)
        :model/type "You must specify a model type 3"})))

#?(:clj
   (defn schema-validation-updated
     [{:keys [ new-delta-model old-delta-model db]} ]     
     (if-let [model-type (:model/type new-delta-model)]       
       (let [new-value (add-delta-model-ref-types
                         {:db db
                          :delta-model new-delta-model})             
             changed-keys (remove
                            #{:model/type :db/id}
                            (into
                              #{} (concat
                                    (keys new-value) (keys old-delta-model))))
             {:keys [req opt]} (get-model-fields model-type)
             errs (into {}
                    (zipmap (remove (into #{} (concat req opt)) changed-keys)
                      (repeat "Is not a valid key")))
             errors (reduce
                      (fn [errs k]                        
                        (let [new-val (get new-value k)
                              val (if (and (nil? new-val) (get req k))
                                    "This is a required key"
                                    (validate-spec-field k new-val))]
                          (if (true? val)
                            errs
                            (assoc errs k val ))))
                      errs changed-keys)]           
         (when (seq errors)
           (assoc errors :db/id (:db/id new-delta-model))))
       {:db/id (:db/id new-delta-model)
        :model/type "You must specify a model type"})))

#?(:clj
   (defn schema-validation
     [{{:keys [db-after] :as tx-report} :tx-report deltas :deltas}]     
     (let [errs (vec
                  (filter
                    identity
                    (map
                      (fn [{:keys [old new]}]                     
                        (cond
                          (and (nil? old) new) (schema-validation-added
                                                 {:delta-model new
                                                  :db db-after})
                          (and old new) (schema-validation-updated
                                          {:old-delta-model old
                                           :new-delta-model new
                                           :db db-after})))
                      deltas)))]       
       (when (seq errs)         
         (throw (ex-info (pr-str "There was a validation error" errs)
                  {:message "There was a schema validation error"
                   :details errs}))))))

#?(:clj
   (defn classify-tx-report [{:keys [tempids tx-data db-before db-after]}]
     (d/q '[:find ?e ?a ?v ?attr ?val ?entity-type ?cardinality ?attr-val-type-name ?added?
            :in $dbb $dba % [[?e ?a ?v ?t ?added?]]
            :where
            ($dbb entity-exists? ?e ?in-before?)
            ($dba entity-exists? ?e ?in-after?)
            ($dba classify-type ?in-before? ?in-after? ?entity-type)            
            [$dba ?a :db/ident ?attr]         
            [$dba ?a :db/valueType ?attr-val-type]
            [$dba ?attr-val-type :db/ident ?attr-val-type-name]
            [$dba ?a :db/cardinality ?c]
            [$dba ?c :db/ident ?cardinality]
            ($dba classify-val ?attr-val-type-name ?v ?val)]
       db-before
       db-after
       '[[(entity-exists? ?e ?exists?)
          [?e]                    
          [(ground true) ?exists?]]
         [(entity-exists? ?e ?exists?)
          (not [?e])                    
          [(ground false) ?exists?]]
         [(classify-type ?in-before? ?in-after? ?type)
          [(= ?in-before? false)]
          [(= ?in-after? true)]          
          [(ground :added) ?type]]
         [(classify-type ?in-before? ?in-after? ?type)
          [(= ?in-before? true)]
          [(= ?in-after? true)]          
          [(ground :updated) ?type]]
         [(classify-type ?in-before? ?in-after? ?type)
          [(= ?in-before? true)]
          [(= ?in-after? false)]          
          [(ground :removed) ?type]]
         [(classify-val ?attr-type ?v ?val)
          [(= ?attr-type :db.type/ref)]
          [?v :db/ident ?val]]
         [(classify-val ?attr-type ?v ?val)
          [(= ?attr-type :db.type/ref)]
          (not [?v :db/ident])
          [(identity ?v) ?val]]
         [(classify-val ?attr-type ?v ?val)
          [(not= :db.type/ref ?attr-type)]
          [(identity ?v) ?val]]]
       tx-data)))

(defn schemafy [model]
  #?(:clj (schemafy-datomic model))
  #?(:cljs (schemafy-as-map model)))

(defn raise-validation-errors [errs]
  (when (seq errs)    
    (throw (ex-info (str "Validation Error" (pr-str errs))
             {:message "There was a validation error"
              :details errs}))))

(defn resolve-model [model models]
  (let [model-name (:name model)]
    (assoc model :fields
      (reduce-kv (fn [m field-name field-value ]
                   (assoc m (if-not (namespace field-name)
                              (keyword (name model-name) (name field-name))
                              field-name)
                     (if (namespace field-name)
                       (get-in models [field-value :fields (keyword (name field-name))])
                       field-value)))
        {}
        (:fields model)))))

(defn model-map [models]
  (let [model-map' (reduce
                     (fn [mm m]
                       (assoc mm (:name m) m))
                     {}
                     models)]    
    (reduce-kv
      (fn [resolved-models model-name model-def]
        (assoc resolved-models model-name (resolve-model model-def model-map')))
      {}
      model-map')))

#?(:clj
   (defn add-audit-to-req-tx-data [{:keys [req-tx-data access-token]} ]
     (conj req-tx-data (cond-> {:db/id (d/tempid :db.part/tx)
                                :model/type :tx-audit/model}
                         access-token (assoc :tx-audit/token (:db/id access-token))
                         (not access-token) (assoc :tx-audit/source :tx-audit.source/system)) )))

#?(:clj
   (s/fdef add-audit-to-req-tx-data
     :args (s/cat
             :args (s/keys
                     :req-un [:datomscript/access-token
                              :datomscript/req-tx-data]) )
     :ret :datomscript/req-tx-data))

#?(:clj
   (defn add-tx-report [{:keys [req-tx-data conn]}]
     (d/with (d/db conn) req-tx-data)))

#?(:clj
   (s/fdef add-tx-report
     :args (s/cat
             :args (s/keys
                     :req-un [:datomscript/conn
                              :datomscript/req-tx-data]) )
     :ret :datomscript/tx-report))

#?(:clj
   (s/fdef add-audit-to-tx-data
     :args (s/cat
             :args (s/keys
                     :req-un [:datomscript/base-params]) )
     :ret (s/keys
            :req-un [:datomscript/base-params
                     :datomscript/tx-report])))

#?(:clj
   (defn resolve-tmp-ids
     [{{:keys [tempids db-after] :as tx-report} :tx-report tmpids :tmpids}]     
     (reduce
       (fn [acc id]        
         (let [db-id (d/resolve-tempid db-after tempids id)
               eid (:resource/eid (d/entity db-after db-id))]           
           (-> acc               
             (assoc-in [:tmp-id->eid id] eid)
             (assoc-in [:db-id->eid db-id] eid)
             (assoc-in [:tmp-id->db-id id] db-id))))
       {:tmp-id->eid {}
        :db-id->eid {}
        :tmp-id->db-id {}}
       tmpids)))

#?(:clj
   (defn resolve-id-namespace [tmpid]
     (-> tmpid first second)))

#?(:clj
   (defn find-tmp-ids [tx-data & [include-tx?]]
     (let [tmpids (atom [])
           _ (walk/postwalk
               (fn [d]
                 (when (= (type d) datomic.db.DbId)                   
                   (let [ns (resolve-id-namespace d)
                         use-id? (if (= ns :db.part/tx)
                                   include-tx?                                  
                                   true)
                         _ (when use-id?
                             (swap! tmpids conj d))]))
                 d)      
               tx-data)]       
       (into #{} @tmpids))))



#?(:clj
   (defn collect-tmp-ids [tx-data]
     (find-tmp-ids tx-data)))

#?(:clj
   (defn db-id->tmp-id
     [{{:keys [tempids db-after] } :tx-report req-tx-data :req-tx-data}]
     (let [tmpids (find-tmp-ids req-tx-data true)]
       (reduce
         (fn [acc tmpid]
           (assoc acc (d/resolve-tempid db-after tempids tmpid) tmpid ))
         {}
         tmpids))))

(defn split-handler [k f]
  (let [ [attr-types delta-types] (map #(if (sequential? %)
                                          %
                                          [%])
                                    k)
        key-val-pairs (for [a attr-types
                            d delta-types]
                        {[a d] [f]})]
    (reduce (fn [acc m] (merge-with vconcat acc m)) {} key-val-pairs)))

(defn merge-handler-map [handler-map]
  (reduce-kv
    (fn [new-handler k f]
      (merge-with
        vconcat
        new-handler
        (if (some vector? k)
          (split-handler k f)
          {k [f]})))
    {}
    handler-map))

(defn gather-handlers [handlers]  
  (reduce
    (fn [acc handler-map]
      (merge-with
        vconcat
        acc
        (merge-handler-map handler-map)))
    {}
    handlers))

#?(:clj
   (defn transact-fn [{:keys [permission-handlers post-tx-handlers] :as handlers } ]     
     (fn [{:keys [req-tx-data conn skip-permissions? access-token]}]       
       (cond-> (assoc handlers
                 :conn conn
                 :req-tx-data req-tx-data
                 :access-token access-token)
         true (#(update-in % [:req-tx-data]
                  (fn [req-tx-data]                    
                    (add-audit-to-req-tx-data {:req-tx-data req-tx-data
                                               :access-token access-token}))))
         true (#(assoc % :tx-report (add-tx-report
                                      (select-keys % [:req-tx-data
                                                      :conn]))))
         true (#(assoc % :entity-classification
                  (classify-entities
                    (:tx-report %)
                    (into #{} (map (fn [[e]] e) (get-in % [:tx-report
                                                           :tx-data]))) )))
         true (#(assoc % :deltas (add-deltas
                                   (select-keys % [:tx-report
                                                   :entity-classification]))))
         true (#(do
                  (schema-validation
                                     (select-keys % [:tx-report
                                                     :deltas]))
                  %))           
         (not skip-permissions?) (#(do                                     
                                     (permission-validation
                                       (select-keys % [:tx-report
                                                       :deltas
                                                       :permission-handlers]))
                                     %))
         true (#(assoc % :db-id->tmp-id (db-id->tmp-id
                                          (select-keys % [:tx-report
                                                          :req-tx-data]))))
         true (#(assoc % :req-tx-data
                  (append-post-tx-data
                    (select-keys % [:tx-report
                                    :db-id->tmp-id
                                    :deltas
                                    :req-tx-data
                                    :post-tx-handlers]))))
         true (#(assoc % :tx-report @(d/transact (:conn %) (:req-tx-data %)) ))
         true (#(assoc % :tmpids (collect-tmp-ids (:req-tx-data %))))
         true (#(merge % (resolve-tmp-ids (select-keys % [:tx-report
                                                          :tmpids]))))))))

(comment
  (s/explain :datomscript/deltas
    [{:old {:model-type :aardvark
            :x :y
            :b :z}
      :new nil}])

  
  ;; om create access token
  ;; Send anonymous data
  ;; Create access token
  ;; Create anonymous user

  ;; Create company using the access token
  ;; The company and identities should be set
  ;; Should allow a claim of an access token to get it set on the company

  ;; Invite a user
  ;; Should actually create the user and the access token for them
  ;; The permission grant should connect the user to the employees

  ;; validate/fn
  ;; db/fn that should validate based on queries
  ;; Should be able to pass query and params
  ;; params should be a list.  Going to apply query using the db.
  ;; Should be an expected value
  ;; If the validation function does not equal the expected value, then it
  ;; will raise the exception (which will be passed into the function)

  ;; tx-data

  ;; transact-fn
  ;; Takes a models map, creates the permissions, schema validation and
  ;; post-tx-handlers
  ;; Returns a function which should take:
  ;; tx-data, conn, access token and a tmpids (map from om id to {:db/id db-id :resource/eid eid})  
  ;; Add the access token and the tx audit before giving to permission-validation
  ;; skip-permissions?
  ;; Will then run the with transaction, place reslt in tx-report of the map
  

  ;; filter-fn

  ;; Should take an access token first

  ;; The first function should examine the entity for a model type
  ;; Reject if no type on it
  ;; Should never be able to get a password hash, so always exclude
  ;; that attribute
  ;; If has type, should dispatch to the permission read function of
  ;; the model type

  ;; the handlers are a list of maps
  ;; Need to be build into a single map
  ;; key-> vector of functions

  ;; permission handlers can return a vector or map of errors

  
  
  (deltas-reducing
    {:handlers
     (post-tx-handlers
       [{:post-tx-handlers {[:* :*] (fn [_ _] [{:db/id :aardvark}])
                            [:* :added] (fn [_ _] [{:db/id :aardvark5}])
                            [:x :*] (fn [_ _] [{:db/id :aardvark6}])}}
        {:post-tx-handlers {[:* :*] (fn [_ _] [{:db/id :aardvark83}])
                            [:y :*] (fn [_ _] [{:db/id :aardvark77}])}}] )
     :deltas [{:old nil
               :new {:db/id 55
                     :model/type :x}}
              {:old nil
               :new {:db/id 55
                     :model/type :y}}]})
  

  (s/instrument #'datomscript.model/add-audit-to-req-tx-data)
  (s/unstrument #'datomscript.model/classify-entities)
  
(let [access-token {}]
    (cond-> {:req-tx-data []}
      true (#(update-in % [:req-tx-data]
               (fn [req-tx-data]
                 (add-audit-to-req-tx-data {:req-tx-data req-tx-data
                                            :access-token access-token} ))))))
  

  )
