(ns datomscript.client.xhr
  (:require [goog.Uri :as uri]
            [goog.structs :as structs]            
            [cognitect.transit :as t]
            [clojure.walk]
            [cljs.reader :as reader]
            [om.tempid :as tempid :refer [TempId]]
            [goog.events :as events]            
            [goog.Uri.QueryData :as query-data])
  (:import [goog.net XhrIo]
           goog.net.EventType
           [goog.events EventType]))

(defn params-to-str [params]  
  (if params
    (-> params
        clj->js
        structs/Map.
        query-data/createFromMap
        .toString)))

(defn add-query-params [url params]
  (if (seq params)
    (str
     url
     "?"
     (params-to-str params))
    url))

(def ^:private meths
  {:get "GET"
   :put "PUT"
   :post "POST"
   :delete "DELETE"})

(def ^:private transit-reader
  (t/reader :json { :handlers {"om/id" (fn [id] (tempid/tempid id))} }))

(deftype TempIdHandler []
  Object
  (tag [_ _] "om/id")
  (rep [_ r] (. r -id))
  (stringRep [_ _] nil))

(def ^:private
  transit-writer
  (t/writer :json
            {:handlers {TempId (TempIdHandler.) }}))

(defn raw-xhr [{:keys [method url data params on-success on-error reader-fn
                       authorization timeout on-complete query-header
                       host accept-type content-type writer-fn]}]  
  (let [headers {"Content-Type" content-type "Accept" accept-type}
        headers (if authorization
                  (assoc headers "Authorization" (str "Bearer " authorization))
                  headers)
        headers (if query-header
                  (assoc headers "x-query" (pr-str query-header))
                  headers)
        xhr (XhrIo.)        
        url-params (add-query-params url (or params {}))]    
    (when on-complete
      (events/listen xhr goog.net.EventType.COMPLETE
                     (fn [e]
                       (on-complete))))

    (when on-success
      (events/listen xhr goog.net.EventType.SUCCESS
                     (fn [e]                
                       (try
                         (on-success (reader-fn (.getResponseText xhr)))
                         (catch js/Error e
                           (.error js/console e)
                           (.error js/console (.getResponseText xhr))
                           (.error js/console (pr-str e))
                               (when on-error
                                 (on-error {:message "There was an unkonwn server error"})))))))
    (events/listen xhr goog.net.EventType.ERROR
      (fn [e]        
        (when on-error
          (try            
            (on-error (reader-fn (.getResponseText xhr)))
            (catch js/Error e
              (on-error {:message "There was an unknown server error"}))))))
    (events/listen xhr goog.net.EventType.TIMEOUT
      (fn [e]        
        (when on-error
          (try
            (on-error {:message "The request timed out"})
            (catch js/Error e
              (on-error {:message "There was an unknown server error"}))))))
    (.setTimeoutInterval xhr (or 10000 timeout))
    (. xhr
       (send url-params (meths method) (when data (writer-fn data))
        (clj->js headers)))))

(defn edn-xhr [params]
  (raw-xhr (assoc params
                  :accept-type "application/edn"
                  :content-type "application/edn"
                  :reader-fn reader/read-string)))

(defn transit-xhr [params]  
  (raw-xhr (assoc params
                  :url (:url params)
                  :accept-type "application/transit+json"
                  :writer-fn (partial t/write transit-writer)
                  :content-type "application/transit+json"
                  :reader-fn (partial t/read transit-reader))))

(defn internal-api-simple
  [{:keys [data url authorization on-complete on-success on-error]}]  
  (transit-xhr {:method :post
                :url url
                :data data
                :authorization authorization
                :on-complete (fn [x]
                               (when on-complete
                                 (on-complete x)))
                :on-success (fn [x]
                              (when on-success
                                (on-success x)))
                :on-error (fn [x]
                            (when on-error
                              (on-error x)))}))
(comment
  (internal-api-simple {:data '[(test/test1 {})]
                        :on-success (fn [x] (.log js/console "success" (pr-str x )))
                        :on-error (fn [x] (.log js/console "error" (pr-str x )))
                        :on-complete (fn [x] (.log js/console "complete" (pr-str x )))})
  )


