(ns datomscript.client.om.subscription
  (:require [datascript.core :as d]
            [om.next.impl.parser :as om.parser]
            [om.next :as om]
            [clojure.data]
            [datomscript.client.om.error]))

(def vconcat (comp vec concat))

(defn match-subscription-update [e]
  (= (:dispatch-key (om.parser/expr->ast e)) 'subscription/update))

(defn get-expr-in-subscription [subscriptions checksum-ident]
  (some
   (fn [[expr components]]     
     (when (= (:key (om.parser/expr->ast expr)) checksum-ident)
       expr))
   subscriptions))

(defn response->query [subscriptions checksum-response]
  (reduce
   (fn [req [checksum-ident _]]
     (let [expr (get-expr-in-subscription subscriptions checksum-ident)]
       (conj req expr)))
   []
   checksum-response))

(defn add-subscription [subscriptions query]  
  (update subscriptions query (fnil inc 0) 1))

(defn remove-subscription [subscriptions query]
  (let [subs (update subscriptions query dec 1)]    
    (if (<= (get subs query) 0)
      (dissoc subs query)
      subs)))

(defn update-subscriptions [subscriptions query]
  "This updates the subscriptions map that is passed into it.  It will increase
  the sub count on added subs, and decrement the count on removed subs."
  (reduce
   (fn [subs q]     
     (let [{:keys [old new]} (:params (om.parser/expr->ast q))]       
       (-> subs
           (#(reduce (fn [acc q]
                       (add-subscription acc q))
                     %
                     new))
           (#(reduce (fn [acc q]
                       (remove-subscription acc q))
                     %
                     old)))))
   subscriptions
   query))

(defn swap-subscription-params [query]
  (->> (filter match-subscription-update query) 
       (mapv
        (fn [q]
          (let [{:keys [old new]} (:params (om.parser/expr->ast q))
                new-params (cond-> {}
                             new (assoc :old new)
                             old (assoc :new old))]
            (om.parser/ast->expr
             (assoc (om.parser/expr->ast q)                    
                    :params new-params)))))))

(defn query-with-error [response query]
  "Finds all the subscriptions with errors in the responses, and returns
  their queries."
  (->> (filter match-subscription-update query)       
       (reduce
        (fn [error-queries e]
          (if (datomscript.client.om.error/contains-error? (get response (:key (om.parser/expr->ast e))))
            (conj error-queries e)
            error-queries))
        [])))

(defn revert [subscriptions query]
  (update-subscriptions
   subscriptions
   (swap-subscription-params query)))

(defn subscription-deltas [old-subscriptions new-subscriptions]
  "Given a list of old and new subscriptions, it will return a single
  subscription/update query with the keys old and new for the params."
  (let [[old new] (clojure.data/diff old-subscriptions new-subscriptions)
        old-keys (keys old)
        new-keys (keys new)
        changed (cond-> {}
                  (seq old-keys) (assoc :old (vec old-keys))
                  (seq new-keys) (assoc :new (vec new-keys)))]
    (when (seq changed)
      `(subscription/update ~changed))))

(defn old-expr-sent [expr old]
  (vec (filter (set old) (-> expr om.parser/expr->ast :params :old))))

(defn old-expr-not-sent [expr old]
  (vec (remove (set old) (-> expr om.parser/expr->ast :params :old))))

(defn new-expr-sent [expr new]
  (vec (filter (set new) (-> expr om.parser/expr->ast :params :new))))

(defn new-expr-not-sent [expr new]
  (vec (remove (set new) (-> expr om.parser/expr->ast :params :new))))

(defn expr-deltas-step [expr old new old-fn new-fn]
  (let [old' (old-fn expr old)
        new' (new-fn expr new)
        updated-params (cond-> {}
                         (seq old') (assoc :old old')
                         (seq new') (assoc :new new'))]
    (when (seq updated-params)
      (om.parser/ast->expr (assoc (om.parser/expr->ast expr) :params updated-params)))))

(defn query-deltas-step [query old new old-fn new-fn]
  (reduce
   (fn [new-query expr]
     (if-let [expr' (expr-deltas-step expr old new old-fn new-fn)]
       (conj new-query expr')
       new-query))
   []
   query))

(defn query-deltas [query subscription-deltas]
  "Determines which queries components were sent, and which were not sent.  sent and not-sent will both be represented as a subscription/update with new and old params."  
  (let [{:keys [old new ]} (:params (om.parser/expr->ast subscription-deltas))]    
    {:sent (query-deltas-step query old new old-expr-sent new-expr-sent  )
     :not-sent (query-deltas-step query old new old-expr-not-sent new-expr-not-sent)}))

(defn process-subscription-query
  "subscription/update expressions are merged into one expression by removing
  redundant subscriptions.  Returns a new query, and subscription map.  The map
  contains which subscriptions were sent, and which ones weren't."
  [subscriptions query]  
  (let [sub-exprs
        ,(vec (filter
               match-subscription-update
               query))        
        new-sub (update-subscriptions subscriptions sub-exprs)        
        sub-expr-delta (subscription-deltas subscriptions new-sub)        
        query-without-subs
        ,(vec (remove
               match-subscription-update
               query))
        {:keys [sent] :as q-deltas} (when sub-expr-delta
                                        (query-deltas sub-exprs sub-expr-delta))
        sent-queries (when sub-expr-delta
                       (get-in (om.parser/expr->ast sub-expr-delta) [:params :new]))]
    {:query (cond-> query-without-subs
              (seq sub-expr-delta) (conj sub-expr-delta)
              (seq sent-queries) (vconcat sent-queries))
     ;; This is basically :new, :sent, :not-sent
     ;; where new is the new subscription map that will be transacted back
     :sub (merge
           {:new new-sub}
           q-deltas)}))
