(ns datomscript.client.om.send
  (:require [datomscript.core.om.parser]
            [om.next.protocols :as om.protocols]
            [datomscript.client.om.db]
            [datomscript.client.om.merge ]
            [datascript.core :as d]
            [datomscript.client.om.subscription]))

(def vconcat (comp vec concat))

(defn update-remote-sync!
  [{:keys [conn subs query reconciler]}]  
  (let [{:keys [tx-data keys]}
        ,(reduce
          (fn [tx-data expr]
            (let [{:keys [on-send-fn on-send-tx]}
                  ,(datomscript.core.om.parser/extract-app-params expr)]
              (merge-with
               vconcat
               tx-data
               (datomscript.client.om.merge/run-handlers
                (datomscript.client.om.merge/gather-handlers on-send-fn on-send-tx)
                {:query expr
                 :db @conn}))))
          {:tx-data []
           :keys []}
          query)         
        tx-data (conj tx-data {:db/id (datomscript.client.om.db/tempid)
                               :app/id :app/subscriptions
                               :app/value (or subs {})})]    
    (when (seq tx-data)
      (d/transact! conn tx-data)
      (om.protocols/queue! reconciler (set keys)))))

(defn process-query!
  [{:keys [query reconciler conn sub-merge-fns]}]
  "This makes sure the roots are processed by om, the subscription map is updated,
  and all of the on-send-fn and on-send-tx functions are ran on each of the queries.
  Can also pass in sub-merge-fns, which will add merge functions to the
  subscription/update queries."
  (let [query (datomscript.core.om.parser/process-roots query)        
        subs (:app/value (d/entity @conn [:app/id :app/subscriptions]))
        {query :query {:keys [sent] subs :new} :sub}
        ,(datomscript.client.om.subscription/process-subscription-query subs query)
        ;; post-query-process
        ;; Basically add the on-error handler to the subscription/update
        ;; This is where you need to go through the queries
        _ (update-remote-sync! {:conn conn
                                :query query
                                :subs subs                                
                                :reconciler reconciler})]    
    query))

(defn send [{:keys [send-map-router access-token sub-merge-fns]}]
  (fn [reconciler send-map merge-callback]    
    (let [conn (get-in reconciler [:config :state])
          token (access-token @conn)]      
      (doseq [[remote query] send-map]        
        (when (seq query)          
          (let [query (process-query!
                       {:query query
                        :conn conn
                        ;; post-query-fn
                        :reconciler reconciler})
                router-fn (get send-map-router remote)]            
            (router-fn (cond-> {:query query
                                :remote-query
                                (-> query
                                    datomscript.core.om.parser/db-ids->om-ids
                                    datomscript.core.om.parser/remove-app-keys-from-query)
                                :merge-callback merge-callback}
                         token (assoc :token token)))))))))
