(ns datomscript.client.om.merge
  (:require [datascript.core :as d]
            [datomscript.client.om.subscription]
            [datomscript.client.datascript :as datom.ds]
            [om.next :as om]
            [om.next.protocols]
            [datomscript.client.om.db]
            [datomscript.client.om.error]
            [datomscript.core.om.parser]
            [om.next.impl.parser :as om.parser]))

(def ^:dynamic *aardvark* 5)

(def vconcat (comp vec concat))

(defn om-merge [acc result]  
  (let [vconcat-keys [:tx-data :keys :pre-tx-fn :post-tx-fn]]    
    (assoc
      (merge-with
        vconcat
        (select-keys acc vconcat-keys)
        (select-keys result vconcat-keys))
      :tx-meta (merge (:tx-meta acc) (:tx-meta result)))))

(defn cache-key [query]
  (second (:key (om.parser/expr->ast query))))

(defn resolve-tmp-id [query tmp-id]
  (let [tx-report @(:tx-report (datomscript.core.om.parser/extract-app-params query))]    
    (d/resolve-tempid (:db-after tx-report) (:tempids tx-report) tmp-id)))

(defn merge-ids [{:keys [response db query]}]
  (reduce-kv
    (fn [acc om-id resource-eid]
      (let [db-id (.-id om-id)
            db-id (if (< db-id 0)
                    (resolve-tmp-id query db-id)
                    db-id)]
        (merge-with
          conj
          acc
          {:tx-data {:db/id db-id
                     :resource/eid resource-eid}
           :keys (if (> db-id 0)
                   [:db/id db-id]
                   [])})))
    {:tx-data []
     :keys []}
    response))

(defn simple-success-fn [app-id]
  {:on-success-fn (fn [{:keys [response]}]
                    {:tx-data [{:db/id (datom.ds/tempid)
                                :app/id app-id
                                :app/data response}]
                     :keys [[:db/id app-id]]})})

(defn simple-complete-tx [app-id]
  {:on-complete-tx {:tx-data [{:db/id  (datom.ds/tempid)
                               :app/id app-id
                               :app/syncing? false}]
                    :keys [[:app/id :app/loading]]}})

(defn simple-send-tx [app-id]
  {:on-send-tx {:tx-data [{:db/id  (datom.ds/tempid)
                           :app/id app-id
                           :app/syncing? true}]
                :keys [[:app/id :app/loading]]}})

(def default-response-handler
  {:on-complete-fn (fn [_])
   :on-success-fn (fn [{:keys [response db query] :as args}]
                    (merge-ids args))
   :on-error-fn (fn [{:keys [response]}]
                  (.error js/console "transact error" (pr-str response)))})

(def pull-handler
  {:on-complete-fn (fn [{:keys [query db response]}]                  
                     (let [ent (d/entity
                                db (:ident (:params (om.parser/expr->ast query))))
                           [db-id exists?] (if-let [id (:db/id ent)]
                                             [id true]
                                             [(datomscript.client.om.db/tempid) false])]                    
                       {:tx-data (cond-> []
                                   exists? (conj {:db/id db-id
                                                  :app/syncing? false}))
                        :keys (cond-> []
                                exists? (conj [:db/id db-id]))}))
   :on-success-fn (fn [{:keys [query db response]}]
                    [(datomscript.core.om.parser/db-ids->tmpids response)])
   :on-error-fn (fn [])})

(def default-query-handler
  {:on-complete-fn (fn [{:keys [query db]}]                  
                  (let [cache-key' (cache-key query)
                        ent (d/entity db [:app/cache cache-key'])]
                    {:tx-data [{:db/id (:db/id ent)
                                :app/syncing? false}]
                     :keys [[:app/cache cache-key']]}))
   :on-send-fn (fn [{:keys [db query]}]                 
                 (let [cache-key' (cache-key query)
                       ent (d/entity db [:app/cache cache-key'])]
                   {:tx-data [{:db/id (datomscript.client.om.db/tempid)
                               :app/cache cache-key'
                               :app/syncing? true}]
                    :keys [[:app/cache cache-key']]}))
   :on-success-fn (fn [{:keys [query db response sub?]}]                 
                 (let [cache-key' (cache-key query)
                       ent (d/entity db [:app/cache cache-key'])]
                   ;; Need to check for app/params on query to see if
                   ;; cache? is false, and data is true.                   
                   [(cond-> {:db/id (:db/id ent)} 
                      (not sub?) (assoc :app/value response)
                      sub? (assoc :app/sub-value response))]))
   :on-error-fn (fn [])})

(def subscription-update-handler
  {:on-complete (fn [])
   :on-success (fn [])
   :on-error (fn [{:keys [response query db]}]
               (let [ent (d/entity db [:app/id :app/subscriptions])
                     subs (:app/value ent)]
                 [{:db/id (:db/id ent)
                   :app/value (datomscript.client.om.subscription/revert subs query)}]))})

(defn concat-response-request [response request]
  "When given a remote response and the request that generated it, need to group
  them together by the same key/ident into a tuple of two elements. 
  `[resp req dispatch-key]`"
  (reduce
   (fn [acc req]
     (conj acc [(get response (:key (om.parser/expr->ast req)))
                req
                (:dispatch-key (om.parser/expr->ast req))]))
   []
   request))

(defn format-handler-result [result]
  "This makes sure that the result always returns a map with the keys
  :tx-data and :keys.  If result is a vector, it is converted into
  a map with result as tx-data."  
  (cond
    (vector? result) {:tx-data result :keys [] :tx-meta {}
                      :pre-tx-fn [] :post-tx-fn []}
    (nil? result) {:tx-data [] :keys [] :tx-meta {}
                   :pre-tx-fn [] :post-tx-fn []}
    :else (cond-> (assoc result
                    :pre-tx-fn (if-let [f (:pre-tx-fn result)]
                                 [f]
                                 [])
                    :post-tx-fn (if-let [f (:post-tx-fn result)]
                                  [f]
                                  []))
            (not (:tx-meta result))
            ,(assoc result :tx-meta {}))))

(defn gather-handlers [f tx]  
  (cond-> (if (vector? f)
            f
            [f])
    tx (conj (fn [_] tx))))

(defn run-handlers [handlers params]  
  (reduce
    (fn [result f]      
      (if-not f
        result
        (om-merge result (format-handler-result (f params)))))
    {:tx-data []
     :tx-meta {}
     :pre-tx-fn []
     :post-tx-fn []
     :keys []}
    handlers))

(defn merge*-step
  "This is a step function.  It is designed to produce a map with the keys 
  tx-data and keys.  `response` is the response to the `query`.  The query
  is expected to keep its handler functions in the params under the app/params key.  
  There are three different types on-success, on-complete and on-error.  They
  can be -fn or -tx.  -tx should be directly insert into the db, and should
  either be a vector of txes, or a map with the keys :txes and :keys.  The functions
  should expect a map with the keys db, response and query. "
  [{:keys [db response query sub?]}]  
  (let [{:keys [on-complete-tx on-complete-fn
                on-success-tx on-success-fn                
                on-error-tx on-error-fn
                tx-report]}
        (datomscript.core.om.parser/extract-app-params query)        
        handler-params (cond-> {:db db                                
                                :sub? sub?
                                :response response
                                :query query}
                         tx-report (assoc :tx-report @tx-report))        
        on-complete-fns (gather-handlers on-complete-fn on-complete-tx)        
        on-success-fns (gather-handlers on-success-fn on-success-tx)        
        on-error-fns (gather-handlers on-error-fn on-error-tx)]    
    (om-merge     
     (run-handlers on-complete-fns handler-params)     
     (if (datomscript.client.om.error/contains-error? response)
       (run-handlers on-error-fns handler-params)
       (run-handlers on-success-fns handler-params)))))

(defn merge*-reduce
  [{:keys [db reconciler response request sub?]}]
  ;; Should check if response is a server error
  ;; If it is, should create an error for each request query
  (reduce    
    (fn [acc [resp req dispatch-key]]      
      (om-merge acc (merge*-step
                      {:db db
                       :sub? sub?
                       :reconciler reconciler
                       :response resp
                       :query req})))
    {:tx-data []
     :tx-meta {}
     :pre-tx-fn []
     :post-tx-fn []
     :keys []}
   (concat-response-request response request)))

(defn format-response [{:keys [response request]}]
  "Checks for a server error on the response.  Will format the response
  so that a response map is created where each key of the query in the
  response map will have the error."  
  (if-let [err (::om/error response)]
    (datomscript.core.om.parser/uniform-response-from-query request err)    
    response))

(defn merge* [reconciler _ response request & [sub?]]  
  (let [conn (get-in reconciler [:config :state])
        db (d/db conn)
        response (format-response {:response response
                                   :request request} )
        {:keys [tx-data tx-meta pre-tx-fn post-tx-fn keys] :as data-r}
        ,(merge*-reduce
           {:db db
            :sub? sub?
            :reconciler reconciler
            :response response
            :request request})        
        keys (vec (filter seq keys))
        run-post-fns #(doseq [f post-tx-fn]
                        (f %))]
    (doseq [f pre-tx-fn]
      (f))    
    (if (seq tx-data)
      (try        
        (run-post-fns
          (if (seq tx-meta)
            (datom.ds/transact! conn tx-data tx-meta)
            (datom.ds/transact! conn tx-data)))
        ;; Should only be reconciling if we know what to reconcile
        (when (seq keys)          
          (om.next.protocols/queue! reconciler keys))        
        (catch js/Error e
          (.error js/console e)))
      (run-post-fns {:db-after db
                     :db-before db
                     :tx-data []}))))
