(ns datomscript.client.om.core
  (:require [om.next :as om]
            [om.next.protocols]
            [clojure.walk :as walk]
            [goog.log :as glog]
            [om.next.protocols :as p]))

(defn set-tmp-query [state key query]
  (assoc-in state [::om/queries key] query))

(defn set-tmp-query! [state key query]
  (swap! state assoc-in [::om/queries key] query))

(defn get-tmp-query [state key]
  (get-in state [::om/queries key]))

(defn get-props-key-ident [this key]
  (-> (om/props this) key ffirst))

(defn get-props-by-key-path
  ([this key]
   (get-in (om/props this) [key (get-props-key-ident this key)]))
  ([this key params-key]
   (get-in (om/props this) [key (get (om/get-params this) params-key) ])))

(defn set-component-query! [component query-map]
  (let [state (get-in (om/get-reconciler component) [:config :state])]    
    (swap! state assoc-in [::om/queries component] query-map)))

(defn get-raw-query [component]
  (let [state @(get-in (om/get-reconciler component) [:config :state])]    
    (vary-meta (get-in state [::om/queries component :query]) dissoc :component)))

(defn set-component-ref! [component ref]  
  (swap!
   (:indexes (om/get-indexer (om/get-reconciler component)))
   update-in
   [:ref->components ref]
   (fnil conj #{})
   component))

(defn set-component-refs! [component refs]  
  (swap!
   (:indexes (om/get-indexer (om/get-reconciler component)))
   update
   :ref->components
   (fn [ref->components]     
     (reduce
      (fn [ref->comps r]
        (update ref->comps r (fnil conj #{}) component))
      ref->components
      refs))))

(defn set-query-params! [component {:keys [params query]}]
  (let [st (get-in (om/get-reconciler component) [:config :state])]
    (swap! st update-in [:om.next/queries component] merge
           (merge (when query {:query query}) (when params {:params params})))))

(defn set-query!
  "Change the query of a component. Takes a map containing :params and/or
   :query. :params should be a map of new bindings and :query should be a query
   expression. Will schedule a re-render as well as remote re-sends if
   necessary."
  ([x params&query]
   (set-query! x params&query nil))
  ([x {:keys [params query]} reads]
   {:pre [(or (om/reconciler? x)
              (om/component? x))
          (or (not (nil? params))
              (not (nil? query)))
          (or (nil? reads)
              (vector? reads))]}
   (let [r    (if (om/component? x)
                (om/get-reconciler x)
                x)
         c    (when (om/component? x) x)
         root (:root @(:state r))
         cfg  (:config r)
         st   (:state cfg)
         id   (random-uuid)
         _    (.add (:history cfg) id @st)]     
     (when-let [l (:logger cfg)]
       (.log js/console (str (when-let [ident (when (implements? om/Ident c)
                                                (om/ident c (om/props c)))]
                               (str (pr-str ident) " "))
                          (when (om/reconciler? x) "reconciler ")
                          (when query "changed query '" query ", ")
                          (when params "changed params " params " ")
                          (pr-str id)))
       #_(glog/info l
                  (str (when-let [ident (when (implements? om/Ident c)
                                          (om/ident c (om/props c)))]
                         (str (pr-str ident) " "))
                       (when (om/reconciler? x) "reconciler ")
                       (when query "changed query '" query ", ")
                       (when params "changed params " params " ")
                       (pr-str id))))
     (swap! st update-in [:om.next/queries (or c root)] merge
            (merge (when query {:query query}) (when params {:params params})))
     (when (and (not (nil? c)) (nil? reads))
       (p/queue! r [c]))
     (when-not (nil? reads)
       (p/queue! r reads))
     (p/reindex! r)
     (let [rootq (om/get-query c)           
           sends (om/gather-sends (om/to-env cfg)
                                  (into (or rootq [])
                                        (om/transform-reads r reads))
                                  (:remotes cfg))]
       (when-not (empty? sends)
         (p/queue-sends! r sends)
         (om/schedule-sends! r)))     
     nil)))

(defn refresh-remote-component! [c]
  (let [r (om/get-reconciler c)
        cfg  (:config r)
        rootq (om/get-query c)        
        sends (om/gather-sends
                (om/to-env cfg)
                (if (map? rootq)
                  [{:->tmp-key rootq}]
                  rootq)
                (:remotes cfg))]    
    (when-not (empty? sends)      
      (p/queue-sends! r sends)
      (om/schedule-sends! r))))

(defn remove-query-key [query key]
  (let [ast (om/query->ast query)]
    (om/ast->query
     (assoc ast :children (vec (filter
                                (fn [child-ast]
                                  (not (= (:dispatch-key child-ast) key)))
                                (:children ast)))))))

(comment
  (remove-query-key [{:asdf [:db/id :x :y]}
                     {:->x [:db/id :x]}]
    :->y)
  
  )
