(ns datomscript.client.datascript
  (:require [clojure.walk :as walk]
            #?@(:cljs
                [[datascript.core :as d]
                 [cognitect.transit :as transit]               
                 [com.cognitect.transit.types]])))

;; The main schema functions
(defmacro fields
  "Simply a helper for converting (fields [name :string :indexed]) into {:fields {\"name\" [:string #{:indexed}]}}"
  [& fielddefs]
  (let [defs (reduce
              (fn [a [nm tp & opts]]
                (let [set-opts (set opts)]
                  (assoc a (name nm)
                         (cond-> {:db/valueType tp}
                           (= tp :ref) (assoc :db/valueType :db.type/ref)                           
                           (get set-opts :component) (assoc :db/isComponent true)
                           (get set-opts :unique-identity) (assoc :db/unique :db.unique/identity)
                           (get set-opts :many) (assoc :db/cardinality :db.cardinality/many)))))
              {}
              fielddefs)]
    `~defs))

(defn schema*
  "Allow"
  [name maps]  
  (reduce-kv
   (fn [s field-name field-map]
     (merge s {(keyword name field-name) field-map}))
   {}
   maps))

(defmacro schema
  [nm & fields]  
  `(schema* ~(name nm) ~@fields))

#?(:cljs
   (deftype DatomHandler []
     Object
     (tag [_ _] "datascript/Datom")
     (rep [_ d] #js [(.-e d) (.-a d) (.-v d) (.-tx d)])
     (stringRep [_ _] nil)))

#?(:cljs
   (def transit-writer
     (transit/writer :json
                     { :handlers
                      { datascript.db/Datom (DatomHandler.)
                       datascript.btset/Iter (transit/VectorHandler.) }})))

#?(:cljs
   (def transit-reader
     (transit/reader :json
                     { :handlers
                      { "datascript/Datom" (fn [[e a v tx]] (d/datom e a v tx)) }})))

#?(:cljs
   (defn db->string [db]
     (transit/write transit-writer (d/datoms db :eavt))))

#?(:cljs
   (defn string->db [schema db-str]
     (let [datoms (transit/read transit-reader db-str)]
       (d/init-db datoms schema))))

#?(:cljs
   (defn persist [db persist-filter-fn loc]
     (let [db-to-store (d/filter db persist-filter-fn)]
       (js/localStorage.setItem loc (db->string db-to-store)))))

#?(:cljs
   (defn -restore [schema loc]
     (when-let [s (js/localStorage.getItem loc)]    
       (string->db schema s))))

#?(:cljs
   (defn restore [conn schema loc]
     (reset! conn (-restore schema loc))))

#?(:cljs
   (defn create-conn
     [{:keys [schema db-string tx-data]}]
     (let [db-schema (walk/postwalk
                      (fn [d]
                        (if (and (map? d) (not= (:db/valueType d) :db.type/ref))
                          (dissoc d :db/valueType)
                          d))
                      schema)
           c (if-let [d (when (and db-string )
                          (string->db db-schema db-string))]
               (atom d :meta {:listeners (atom {}) })
               (d/create-conn
                (reduce-kv
                 (fn [s k v]
                   (assoc s k (if (not= (:db/valueType v) :db.type/ref)
                                (dissoc v :db/valueType)
                                v)))
                 {}
                 db-schema)))]
       (when (seq tx-data)
         (d/transact c tx-data))
       (alter-meta! c assoc :schema schema)
       c)))

#?(:cljs
   (defn extract-schema [conn]
     (:schema (meta conn))))

#?(:cljs
   (defn tempid
     ([]
      (d/tempid :db.part/user))
     ([id]
      (d/tempid :db.part/user id))))

#?(:cljs
   (defn transact! [conn tx-data & [tx-meta]]
     "A transact! wrapper, mostly for making sure errors
  are properly logged--at least to console."
     (let [transact! (partial d/transact! conn tx-data)]
       (try         
         (if tx-meta
           (transact! tx-meta)
           (transact!))
         (catch js/Error e           
           (.error js/console e)
           (throw e))))))

#?(:cljs
   (defn tempid? [n]
     (if n
       (neg? n)
       false)))

#?(:cljs
   (defn resolve-id [db tempids id]
     (if (tempid? id)
       (d/resolve-tempid db tempids id)
       id)))

(comment
  
  resource
  eid
  uri
  credentials
  db/id Either number or an om tempid

  (require 'om.tempid)

  (map? (om.tempid/tempid 5))
  
  (schemafy-model ^{:unique [:d :f]
                    :component [:e]} {:x sc/Str
                                      (sc/optional-key :y) sc/Str
                                      (sc/required-key :z) sc/Str
                                      :a {:x 5}
                                      :b [{:x 5}]
                                      :c [sc/Str]
                                      :d sc/Str
                                      :e {:x 5}
                                      :f {:x 5}})
  )
