(ns datomscript.api-dev.server.response
  (:require [io.pedestal.test :refer [response-for]]            
            [cognitect.transit :as transit]
            [clojure.tools.logging :as log]
            [datomscript.api.middleware]
            [clojure.java.io :refer [input-stream]]            
            [net.cgrand.xml :as xml]
            [cheshire.core :refer [generate-string parse-string]])
  (:import [java.io ByteArrayInputStream ByteArrayOutputStream InputStreamReader BufferedReader]
           [org.xml.sax.InputSource]))

;;;;;;;;;;;;;;;;;; XML
(defn from-xml [s]
  (xml/parse (org.xml.sax.InputSource. (InputStreamReader. (ByteArrayInputStream. (.getBytes s))))))

;;;;;;;;;;;;;;;;;; TRANSIT
(defn to-transit [m]  
  (let [out (ByteArrayOutputStream. 4096)
        writer (transit/writer
                out :json
                {:handlers datomscript.api.middleware/transit-write-handlers})]
    (transit/write writer m)    
    (.toString out )))

(defn read-transit [t]
  (transit/read
   (transit/reader
    (ByteArrayInputStream.
     (.getBytes t)) :json {:handlers datomscript.api.middleware/transit-read-handlers})))

(def transit-headers
  {"Content-Type" "application/transit+json"
   "Accept" "application/transit+json"})

(def text-headers
  {"Content-Type" "text/plain"
   "Accept" "text/plain"})

(def twillo-headers
  {"Content-Type" "application/json"
   "Accept" "text/xml"})

(defn -response-for [server base-headers body-converter verb url & body-headers]  
  
  (let [[body headers] (if (= verb :get)
                         ["" (first body-headers)]
                         [(-> body-headers first body-converter)
                          (second body-headers)])
        headers (merge base-headers headers)]    
    (response-for server
                  verb url
                  :body body
                  :headers headers)))


(defn basic-response-for  
  [server base-headers from-conversion to-conversion verb url & body-headers]
  (let [r (apply -response-for server base-headers to-conversion verb url body-headers)]    
    (assoc r :body (from-conversion (:body r)))))

(defn body-response-for
  [server base-headers from-conversion to-conversion verb url & body-headers]  
  (:body (apply basic-response-for server base-headers from-conversion to-conversion verb url body-headers)))

(defn transit-response-fns [server]
  {:resp-for (partial body-response-for server transit-headers read-transit to-transit)
   :raw-resp-for (partial basic-response-for server transit-headers read-transit to-transit)})

(defn text-response-fns [server]
  {:resp-for (partial body-response-for server text-headers identity identity)
   :raw-resp-for (partial basic-response-for server text-headers identity identity)})

(defn twillo-response-fns [server]
  {:resp-for (partial body-response-for server twillo-headers from-xml generate-string)
   :raw-resp-for (partial basic-response-for server twillo-headers from-xml generate-string)})

(defn event-response-for   
  [server base-headers from-conversion to-conversion verb url & body-headers]
  (let [r (apply -response-for server base-headers to-conversion verb url body-headers)]    
    ""))

(defn event-response-fns [server]
  {:resp-for (partial event-response-for server text-headers identity identity)})


