(ns ui-components.forms.new-course
  (:require [reagent.core :as r]
            [shared.models.query.index :as query]
            [shared.protocols.actionable :as ac]
            [shared.protocols.specced :as sp]
            [ui-components.forms.sections.checkpoints :refer [CheckpointsSection]]
            [ui-components.forms.sections.controls :refer [FormControls]]
            [ui-components.forms.sections.goal :refer [GoalSection]]
            [ui-components.forms.utils :refer [debounce submit-form]]
            [ui-components.overlay :refer [Overlay]]
            [ui-components.semantic :refer [Form Message]]
            [shared.protocols.loggable :as log]))

(defn create-checkpoint [local-state]
  (swap! local-state #(ac/perform %1 [:add :new-checkpoint])))

(defn update-checkpoint [local-state checkpoint]
  (swap! local-state #(ac/perform %1 [:update checkpoint])))

(defn update-goal [local-state new-goal]
  (swap! local-state assoc :goal new-goal))

(def notifications
  {:existing-goal {:type :error
                   :message "You already have a course with this title!"}
   :not-logged-in {:type :warning
                   :message "You are not logged in yet..."}})

(defn NewCourseForm [{:keys [draft on-cancel on-change on-submit]}]
  (let [local-state       (r/atom draft)
        update-goal       (comp (debounce on-change 1000) update-goal)
        update-checkpoint (comp (debounce on-change 1000) update-checkpoint)]
    (fn [{:keys [flags draft]}]
      (let [{:keys [goal checkpoints] :as course} @local-state
            {:keys [saveable? checking?]}         flags
            {:keys [invalid-fields errors]} (meta draft)]
        [Overlay {:title "Create Course"
                  :size  :large}
         [Form {:error    (not (empty? errors))
                :onSubmit #(submit-form %1 course on-submit)}
          [Message {:error true} (when-not (empty? errors) (:message ((-> errors first :reason) notifications)))]
          [GoalSection {:goal     goal
                        :invalid? (contains? invalid-fields :goal)
                        :check    #(on-change course)
                        :update   (partial update-goal local-state)}]
          [CheckpointsSection {:create-checkpoint #(create-checkpoint local-state)
                               :update-checkpoint (partial update-checkpoint local-state)
                               :checkpoints       checkpoints}]]
         [FormControls {:on-cancel  on-cancel
                        :on-confirm #(on-submit course)
                        :loading?   checking?
                        :disabled?  (not saveable?)}]]))))
