(ns ui-components.forms.new-course
  (:require [reagent.core :as r]
            [shared.models.query.index :as query]
            [shared.protocols.actionable :as ac]
            [shared.protocols.specced :as sp]
            [ui-components.forms.sections.checkpoints :refer [CheckpointsSection]]
            [ui-components.forms.sections.controls :refer [FormControls]]
            [ui-components.forms.sections.goal :refer [GoalSection]]
            [ui-components.forms.utils :refer [debounce submit-form]]
            [ui-components.overlay :refer [Overlay]]
            [ui-components.semantic :refer [Form Message]]
            [shared.protocols.loggable :as log]))

(defn create-checkpoint [local-state]
  (swap! local-state #(ac/perform %1 [:add :new-checkpoint])))

(defn update-checkpoint [local-state checkpoint]
  (swap! local-state #(ac/perform %1 [:update checkpoint])))

(defn update-goal [local-state new-goal]
  (swap! local-state assoc :goal new-goal))

(defn NewCourseForm [{:keys [course on-cancel on-change on-submit]}]
  (let [local-state (r/atom course)
        update-goal (comp (debounce on-change 1000) update-goal)]
    (fn [{:keys [flags errors]}]
      (let [{:keys [goal checkpoints] :as course} @local-state
            {:keys [saveable? checking?]} flags
            app-errors        errors
            valid?            (sp/valid? course)
            saveable?         (and valid? saveable?)
            errors            (sp/errors course)
            invalid-fields    (into #{} (mapcat :in (-> errors first second)))]
        [Overlay {:title "Create Course"
                  :size :large}
         [Form {:error (boolean errors)
                :onSubmit #(submit-form %1 course on-submit)}
          [Message {:error true} (:message (first app-errors))]
          [GoalSection {:goal goal
                        :invalid? (contains? invalid-fields :goal)
                        :check   #(on-change course)
                        :update  (partial update-goal local-state)}]
          [CheckpointsSection {:create-checkpoint #(create-checkpoint local-state)
                               :update-checkpoint (partial update-checkpoint local-state)
                               :checkpoints checkpoints}]]
         [FormControls {:on-cancel on-cancel
                        :on-confirm #(on-submit course)
                        :loading? checking?
                        :disabled? (not saveable?)}]]))))
