(ns backend-shared.protocol_extensions.convertible
  (:require [shared.protocols.convertible :as cv :refer [Convertible]]
            [clojure.string :as str]
            [cljs.spec :as spec]))

(defn res [code body]
  (-> {:statusCode code
       :headers {:Access-Control-Allow-Origin "*"}
       :body (when body (if (spec/valid? keyword? body)
                          (name body)
                          (cv/to-json body)))}
      (with-meta {:spec :offcourse/response})))

(extend-protocol Convertible
  nil
  (-to-clj [_]                  nil)
  array
  (-to-clj [js-arr]             (js->clj js-arr :keywordize-keys true))
  object
  (-to-clj [js-obj]             (js->clj js-obj :keywordize-keys true))
  (-to-json     [obj]           (->> obj clj->js (.stringify js/JSON)))
  string
  (-to-clj [string]             (->> string (.parse js/JSON) cv/to-clj))
  (-to-query [string]
    (let [[repo curator course-id revision checkpoint-id] (str/split string "::")]
      {:course-id (str repo "::" curator "::" course-id)
       :revision (int revision)}))
  PersistentHashMap
  (-to-json     [obj]           (->> obj clj->js (.stringify js/JSON)))
  PersistentArrayMap
  (-to-json     [obj]           (->> obj clj->js (.stringify js/JSON)))
  LazySeq
  (-to-json     [obj]           (->> obj clj->js (.stringify js/JSON)))
  PersistentVector
  (-to-json     [obj]           (->> obj clj->js (.stringify js/JSON)))
  (-to-api      [[res-code message]] (case res-code
                                       :reject    (res 401 message)
                                       :error     (res 500 message)
                                       :success   (res 201 nil)
                                       :found     (res 200 message)
                                       :not-found (res 404 message)
                                       :reply     (res 200 message))))
