(ns backend-shared.service.index
  (:require [backend-shared.aws-event.index :as aws-event]
            [backend-shared.specs.index]
            [backend-shared.protocol-extensions.index]
            [backend-adapters.index :as adapters]
            [backend-shared.service.log :as log]
            [backend-shared.service.fetch :as fetch]
            [backend-shared.service.perform :as perform]
            [backend-shared.service.check :as check]
            [backend-shared.service.respond.index :as respond]
            [shared.protocols.convertible :as cv]
            [shared.protocols.actionable :refer [Actionable]]
            [shared.protocols.eventful :refer [Eventful]]
            [shared.protocols.queryable :refer [Queryable]]
            [shared.models.payload.index :as payload]
            [cljs.spec.alpha :as spec]
            [shared.models.error.index :as error]
            [clojure.string :as str]))

(defrecord Service []
  Actionable
  (-perform [service action] (perform/perform service action))
  (-check   [service action] (check/check service action))
  Eventful
  (-respond [service payload] (respond/respond service payload))
  Queryable
  (-fetch [service query] (fetch/fetch service query)))

(defn resolve-user [raw-event]
  (when-let [auth-id (-> raw-event :requestContext :authorizer :principalId)]
    (let [[provider user-name] (str/split auth-id "|")
          user (when (= "offcourse" provider) {:user-name user-name})
          guest (when-not (= "offcourse" provider) {:auth-id auth-id})]
      (or user guest))))

(defn create [config args]
  (let [[event context callback] args
        {:keys [is_endpoint? specs name adapters]} config]
    (log/incoming event context)
    (let [adapters (adapters/create adapters)
          errors   (keep :error adapters)
          event (cv/to-clj event)
          config {:service-name name
                  :stage        (.. js/process -env -STAGE)
                  :specs        specs
                  :callback     callback
                  :is_endpoint? is_endpoint?
                  :user         (resolve-user event)
                  :context      (cv/to-clj context)
                  :event        event}]
      (map->Service (merge config adapters)))))

(def fetch fetch/fetch)
(def perform perform/perform)
(def check check/check)
