(ns backend-adapters.es.impl.save
  (:require [cljs.core.async :refer [chan close! put!]]
            [cljs.nodejs :as node]
            [shared.protocols.convertible :as cv]
            [shared.protocols.loggable :as log]
            [shared.models.event.index :as event]
            [shared.models.error.index :as error]))

(def AWS (node/require "aws-sdk"))
(def path (node/require "path"))
(def creds (AWS.EnvironmentCredentials. "AWS"))
(def HTTP (AWS.NodeHttpClient.))

(defn create-request [endpoint {:keys [id item index-name] :as q}]
  (let [req (AWS.HttpRequest. endpoint)
        access-key-id (.-accessKeyId creds)
        session-token (.-sessionToken creds)
        headers (aget req "headers")]
      (aset req "method" "POST")
      (aset req "path"  (.join path "/" "offcourse" (name index-name) id))
      (aset req "region" "us-east-1")
      (aset headers "presigned-expires" false)
      (aset headers "Host" (aget endpoint "host"))
      (aset req "body" (.stringify js/JSON (clj->js item)))
      (when (and access-key-id session-token) (.addAuthorization (AWS.Signers.V4. req "es") creds (js/Date.)))
      req))

(defn handle-error [error]
  (log/log "error" (clj->js (event/create [:failed (error/create :http-error error)])))
  (event/create [:failed (error/create :http-error error)]))

(defn handle-success [res]
  (let [statusCode (aget res "statusCode")
        body       (aget res "body")]
    (cond
      (< statusCode 400) (event/create [:found (if (string? body)
                                                 (->> body (.parse js/JSON) cv/to-clj)
                                                 (.toString body "base64"))])
      (= statusCode 404) (event/create [:not-found res])
      :default (event/create [:failed (error/create :http-error res)]))))

(defn handle-response [c error res body]
  (log/log (if error "X" "H"))
  (if error
    (put! c (handle-error error))
    (put! c (handle-success res))))

(defn save [instance {:keys [id item index-name] :as query} endpoint]
  (let [opts {:url (.join path endpoint "/" "offcourse" (name index-name) id)
              :body (clj->js item)}
        c (chan)]
    (.post instance (clj->js (merge query {:url (.join path "/" "offcourse" (name index-name) id)})) #(handle-response c %1 %2 %3))
    c))
