(ns backend-shared.converters.index
  (:require [shared.models.course.index :as course]
            [cljs.nodejs :as node]
            [backend-shared.converters.mercury :as mercury]
            [shared.models.checkpoint.index :as checkpoint]
            [shared.protocols.convertible :as cv]
            [clojure.string :as str]
            [cljs.spec.alpha :as spec]
            [shared.protocols.loggable :as log]
            [clojure.walk :as walk]))

(def to-text (node/require "html-to-text"))

(def atob (node/require "atob"))
(def yaml (node/require "js-yaml"))
(def Zip (node/require "node-zip"))

(def zip-key :data)

(defn yaml->json [input]
  (.safeLoad yaml input))

(defn clj->compressed [payload]
  (-> (new Zip)
      (.file (name zip-key) (.stringify js/JSON (clj->js payload)))
      (.generate (clj->js {:base64 true :compression :DEFLATE}))))

(defn compressed->clj [payload]
  (-> (new Zip (:compressed-data payload) (clj->js {:base64 true :checkCRC32 true}))
       cv/to-clj
       (get-in [:files zip-key :_data])
       cv/to-clj))

(defn raw->profile [{:keys [profile auth-profile]}]
  (-> auth-profile
      (dissoc :identities :picture)
      (assoc :user-name (:user-name profile)
             :revision 1)))

(defn raw->identity [{:keys [provider user_id]} user-name]
  {:auth-id (str provider "|" user_id)
   :user-name user-name})

(defn raw->identities [{:keys [auth-profile profile]}]
  (map #(raw->identity %1 (:user-name profile)) (:identities auth-profile)))

(defn raw->portrait [{:keys [profile auth-profile]}]
  {:portrait-url (str (:picture auth-profile) "&s=300")
   :user-name (:user-name profile)})

(defn find-checkpoint-index [checkpoints checkpoint-id]
  (->> checkpoints
       (map-indexed vector)
       (filter #(= (:checkpoint-id (second %)) checkpoint-id))
       first
       first))

(defn course->update [{:keys [source course resource] :as bookmark-data}]
  (let [{:keys [checkpoint-id]} source
        {:keys [checkpoints]} course
        {:keys [resource-url tags]} resource]
    (assoc source
           :resource-url resource-url
           :checkpoint-index (find-checkpoint-index checkpoints checkpoint-id)
           :tags (take 3 (into #{} tags)))))

(defn yaml-file? [{:keys [path] :as ref}]
  (re-find #"\.yaml$" path))

(defn repo->items [{:keys [curator repo]}]
  (->> (:tree repo)
       (filterv yaml-file?)
       (map #(assoc %1 :curator curator))))
(defn repo-item->course [{:keys [repo-item curator] :as repo-item}]
  (-> (:content repo-item)
      atob
      yaml->json
      js->clj
      walk/keywordize-keys
      (assoc :curator curator)
      course/initialize))

(defn raw->user-data [raw-user]
  {:profile (raw->profile raw-user)
   :portrait (raw->portrait raw-user)
   :identities (raw->identities raw-user)})

(defn course->bookmarks [course]
  (-> course
      course/create
      cv/to-bookmarks))

(defn res [code body]
  (-> {:statusCode code
       :headers {:Access-Control-Allow-Origin "*"}
       :body (when body (if (spec/valid? keyword? body)
                          (name body)
                          (cv/to-json body)))}
      (with-meta {:spec :offcourse/response})))

(defn response->api [[res-code message]]
  (case res-code
    :rejected  (res 401 message)
    :not-found (res 404 message)
    :failed    (res 500 message)
    (res 200 message)))

(def mercury->resource mercury/->resource)
