(ns backend-adapters.s3.converters.action
  (:require [backend-shared.crypto :as crypto]))

(defn ->payload [bucket-name {:keys [key body]}]
  {:Bucket bucket-name
   :Key key
   :Body (.stringify js/JSON body)})

(defn raw-user->payload [user]
  {:key (-> user :profile :user-name)
   :body (.stringify js/JSON user)})

(defn portrait->payload [{:keys [user-name portrait-data]}]
  {:key (str "portraits/" user-name ".jpg")
   :body (js/Buffer. portrait-data "base64")})

(defn bookmark->payload [{:keys [timestamp resource-url] :as bookmark}]
  {:key (str timestamp "--" (crypto/hasher resource-url))
   :body bookmark})

(defn augmented-bookmark->payload [{:keys [timestamp resource-url] :as bookmark}]
  {:key (str "bookmarks/" timestamp "--" (crypto/hasher resource-url))
   :body bookmark})

(defn repo-item->payload [{:keys [sha] :as item}]
  {:key (str "repo-items/" sha)
   :body item})

(defn resource->payload [{:keys [resource-url] :as resource}]
  {:key (str (crypto/hasher resource-url) ".json")
   :body resource})

(defn raw-resource->payload [{:keys [url] :as resource}]
  {:key (str (crypto/hasher url) ".json")
   :body resource})

(def converters {:raw-user raw-user->payload
                 :portrait portrait->payload
                 :raw-resource raw-resource->payload
                 :resource resource->payload
                 :augmented-bookmark augmented-bookmark->payload
                 :bookmarks bookmark->payload
                 :repo-items repo-item->payload})

(defn converter [bucket-names payload-type payload]
  (->payload (payload-type bucket-names)
             ((payload-type converters) payload)))
