(ns backend-shared.converters.index
  (:require [shared.models.course.index :as course]
            [cljs.nodejs :as node]
            [backend-shared.converters.helpers :as h]
            [shared.models.checkpoint.index :as checkpoint]
            [shared.protocols.convertible :as cv]
            [clojure.string :as str]
            [cljs.spec.alpha :as spec]
            [shared.protocols.loggable :as log]))

(def to-text (node/require "html-to-text"))

(def Zip (node/require "node-zip"))
(def zip-key :data)

(defn clj->compressed [payload]
  (-> (new Zip)
      (.file (name zip-key) (.stringify js/JSON (clj->js payload)))
      (.generate (clj->js {:base64 true :compression :DEFLATE}))))

(defn compressed->clj [payload]
  (-> (new Zip (:compressed-data payload) (clj->js {:base64 true :checkCRC32 true}))
       cv/to-clj
       (get-in [:files zip-key :_data])
       cv/to-clj))

(defn raw->profile [{:keys [profile auth-profile]}]
  (-> auth-profile
      (dissoc :identities :picture)
      (assoc :user-name (:user-name profile)
             :revision 1)))

(defn raw->identity [{:keys [provider user_id]} user-name]
  {:auth-id (str provider "|" user_id)
   :user-name user-name})

(defn raw->identities [{:keys [auth-profile profile]}]
  (map #(raw->identity %1 (:user-name profile)) (:identities auth-profile)))

(defn raw->portrait [{:keys [profile auth-profile]}]
  {:portrait-url (str (:picture auth-profile) "&s=300")
   :user-name (:user-name profile)})

(defn course->update [{:keys [source course resource] :as bookmark-data}]
  (let [{:keys [checkpoint-id]} source
        {:keys [checkpoints]} course
        {:keys [resource-url tags]} resource]
    (assoc source
           :resource-url resource-url
           :checkpoint-index (h/find-checkpoint-index checkpoints checkpoint-id)
           :tags (take 3 (into #{} tags)))))

(defn raw->user-data [raw-user]
  {:profile (raw->profile raw-user)
   :portrait (raw->portrait raw-user)
   :identities (raw->identities raw-user)})

(defn course->bookmarks [course]
  (-> course
      course/create
      cv/to-bookmarks))

(defn html->text [content]
  (.fromString to-text content (clj->js {:word-wrap false})))

(defn text->tags [content]
  (-> content
      h/collect-tags
      h/filter-tags))

(defn html->tags [html]
  (-> html
      html->text
      text->tags))

(defn url->video-content [url]
  (let [parsed-url (h/parse-url url)
        id (second (str/split (:query parsed-url) "="))]
    {:provider (:provider parsed-url)
     :id id}))

(defn res [code body]
  (-> {:statusCode code
       :headers {:Access-Control-Allow-Origin "*"}
       :body (when body (if (spec/valid? keyword? body)
                          (name body)
                          (cv/to-json body)))}
      (with-meta {:spec :offcourse/response})))

(defn response->api [[res-code message]]
  (case res-code
    :rejected  (res 401 message)
    :not-found (res 404 message)
    :failed    (res 500 message)
    (res 200 message)))
