(ns backend-shared.service.respond
  (:require [shared.protocols.specced :as sp]
            [shared.protocols.loggable :as log]
            [cljs.spec.alpha :as spec]
            [shared.protocols.convertible :as cv]
            [shared.models.payload.index :as payload]
            [shared.models.error.index :as error]))

(defn res [code body]
  (-> {:statusCode code
       :headers {:Access-Control-Allow-Origin "*"}
       :body (when body (if (spec/valid? keyword? body)
                          (name body)
                          (cv/to-json body)))}
      (with-meta {:spec :offcourse/response})))

(defn to-api [[res-code message]]
  (case res-code
    :rejected  (res 401 message)
    :not-found (res 404 message)
    :failed    (res 500 message)
    (res 200 message)))

(defmulti respond (fn [_ [_ payload]] (sp/resolve (payload/create payload))))

(defmethod respond :error [{:keys [callback is_endpoint? event] :as service} [event-type error]]
  (let [error (merge error {:service-name  (:service-name service)
                            :stage         (:stage service)
                            :event     event})]
    (if is_endpoint?
      (callback nil (clj->js (cv/to-api [event-type error])))
      (callback (.stringify js/JSON (clj->js error))))))

(defmethod respond :invalid-payload [{:keys [callback is_endpoint? event] :as service} [event-type payload]]
  (let [error (-> (error/create :invalid-payload payload)
                  (merge {:service-name  (:service-name service)
                          :stage         (:stage service)
                          :event         event}))]
    (if is_endpoint?
      (callback nil (clj->js (cv/to-api [event-type error])))
      (callback (.stringify js/JSON (clj->js error))))))

(defmethod respond :default [{:keys [callback is_endpoint?]} [_ payload :as event]]
  (if is_endpoint?
    (callback nil (clj->js (cv/to-api event)))
    (callback nil (clj->js payload))))
