(ns backend-adapters.dynamodb.perform
  (:require [backend-adapters.dynamodb.impl :as impl]
            [cljs.core.async :as async]
            [shared.models.error.index :as error]
            [shared.models.payload.index :as payload]
            [shared.protocols.convertible :as cv]
            [shared.protocols.specced :as sp])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defmulti perform (fn [_ [action-type _]] action-type))

(defmethod perform :update [{:keys [table-names] :as this} [_ payload]]
  (go
    (let [{:keys [course-id revision checkpoint-index tags resource-url]} payload
          payload-type (sp/resolve payload)
          table-name (payload-type table-names)
          delta {:TableName table-name
                 :Key {:course-id course-id
                       :revision revision}
                 :UpdateExpression (str "SET #cp[" checkpoint-index "].#tf = :t, #cp[" checkpoint-index "].#uf = :u")
                 :ExpressionAttributeNames {"#cp" "checkpoints"
                                            "#tf" "tags"
                                            "#uf" "resource-url"}
                 :ExpressionAttributeValues {":t" tags
                                             ":u" resource-url}
                 :ReturnValues "ALL_NEW"}
          res (async/<! (impl/update this delta))]
      res)))

(defmethod perform :save [{:keys [table-names] :as this} [_ payload]]
  (go
    (let [payload       (payload/create (into [] payload))
          queries       (cv/to-db payload table-names)
          payload-chans (async/merge (map #(impl/put this %) queries))
          res           (async/<! (async/into [] payload-chans))
          errors      (filter (fn [[result data]] (= :failed result)) res)]
      (if (empty? errors)
        [:success (into [] payload)]
        [:failed (error/create :dynamodb-error (map second errors))]))))
