(ns backend-adapters.s3.to-item
  (:require [clojure.string :as str]
            [shared.models.payload.index :as payload]
            [shared.protocols.convertible :as cv]
            [shared.protocols.specced :as sp]
            [shared.protocols.loggable :as log]
            [backend-shared.crypto :as crypto]))

(defn create-item [item-key bucket-name item-data]
  {:Bucket bucket-name
   :Key item-key
   :Body item-data})

(defmulti to-item (fn [payload payload-type bucket-name] payload-type))

(defmethod to-item :files [{:keys [file-name content]} _ bucket-name]
  (create-item file-name bucket-name content))

(defmethod to-item :augmented-bookmarks [{:keys [timestamp resource-url resource] :as bookmark} _ bucket-name]
  (let [key (str "bookmarks/" timestamp "--" (crypto/hasher resource-url))]
    (create-item key bucket-name (cv/to-json bookmark))))

(defmethod to-item :bookmarks [{:keys [timestamp resource-url] :as bookmark} _ bucket-name]
  (let [key (str timestamp "--" (crypto/hasher resource-url))]
    (create-item key bucket-name (cv/to-json bookmark))))

(defmethod to-item :portraits [{:keys [user-name portrait-data]} _ bucket-name]
  (let [key (str "portraits/" user-name ".jpg")]
    (create-item key bucket-name (js/Buffer. portrait-data "base64"))))

(defmethod to-item :raw-resources [{:keys [url] :as resource} _ bucket-name]
  (create-item (str (crypto/hasher url) ".json") bucket-name (cv/to-json resource)))

(defmethod to-item :resources [{:keys [resource-url] :as resource} _ bucket-name]
  (create-item (str (crypto/hasher resource-url) ".json") bucket-name (cv/to-json resource)))

(defmethod to-item :courses [{:keys [course-id] :as course} _ bucket-name]
  (create-item (str "courses/" course-id) bucket-name (cv/to-json course)))

(defmethod to-item :repo-items [{:keys [sha] :as item} _ bucket-name]
  (create-item (str "repo-items/" sha) bucket-name (.stringify js/JSON (clj->js item))))

(defmethod to-item :raw-users [{:keys [user-name] :as user} _ bucket-name]
  (create-item user-name bucket-name (.stringify js/JSON (clj->js user))))
