(ns backend-shared.models.payload.to-resource
  (:require [clojure.string :as str]
            [shared.models.resource.index :as rs]
            [shared.protocols.specced :as sp]
            [cljs.nodejs :as node]
            [shared.protocols.loggable :as log]))

(def nlp (node/require "compromise"))
(def to-markdown (node/require "to-markdown"))
(def to-text (node/require "html-to-text"))

(defn html->md [content]
  (to-markdown content (clj->js {:gfm true
                                 :converters [{:filter :div
                                               :replacement #(str %1)}]})))

(defn html->text [content]
  (.fromString to-text content (clj->js {:word-wrap false})))

(defn valid-tag? [tag]
  (when-not (re-find #"[\.\.\.]|[(.*)]" tag)
    true))

(defn create-tag [{:keys [name] :as raw-tag}]
  (->> (str/split name " ")
       (map str/lower-case)
       (filter valid-tag?)
       (into #{})
       (take 3)
       (str/join "-")))

(defn create-tags [content]
  (let [all-tags (-> content
                     clj->js
                     nlp
                     .nouns
                     (.out "array"))]
    (->> all-tags
         (filter #(< (count %) 20))
         frequencies
         (sort-by second)
         reverse
         (take 5)
         (map first)
         (into #{}))))


(defmulti to-resource #(sp/resolve %1))

(defmethod to-resource :mercury-resource [{:keys [url title excerpt content]}]
  (let [text-content (html->text content)
        tags         (if (> (count text-content) 200) (create-tags text-content) #{})
        record {:resource-url  url
                :title         title
                :resource-type "html"
                :content       content
                :description   excerpt
                :tags          tags}]
    (rs/create (into {} (remove (comp nil? second) record)))))

(defmethod to-resource :embedly-resource [{:keys [original_url url type title description content
                                               entities language keywords]}]
    (when-not (= "error" type)
      (let [record {:resource-url  url
                    :bookmark-url  (when (not= original_url url) original_url)
                    :title         title
                    :resource-type type
                    :content       content
                    :language      language
                    :description   description
                    :entities      (->> entities (map :name))
                    :tags          (into #{} (map create-tag keywords))}]
        (rs/create (into {} (remove (comp nil? second) record))))))
