(ns backend-adapters.stepfunctions.index
  (:require [cljs.nodejs :as node]
            [cljs.core.async :refer [<! put! close! chan]]
            [shared.protocols.actionable :refer [Actionable]]
            [shared.protocols.loggable :as log]
            [shared.protocols.specced :as sp]
            [shared.models.payload.index :as payload])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def AWS (node/require "aws-sdk"))

(defmulti to-item (fn [payload payload-type machine-name] payload-type))

(defmethod to-item :bookmarks [payload payload-type machine-name]
  {:stateMachineArn machine-name
   :input (.stringify js/JSON (clj->js (first payload)))
   :name (str "api-" (name payload-type) "-" (.now js/Date))})

(defmethod to-item :default [payload payload-type machine-name]
  {:stateMachineArn machine-name
   :input (.stringify js/JSON (clj->js payload))
   :name (str "api-" (name payload-type) "-" (.now js/Date))})

(defn to-payload [payload machine-names]
  (let [payload-type (sp/resolve (payload/create payload))
        machine-name (payload-type machine-names)]
    (to-item payload payload-type machine-name)))

(defn -execute [{:keys [instance machine-names]} query]
  (let [c     (chan)]
    (.startExecution instance (clj->js query)
                     #(let [response (if %1 [:error %1] [:success %2])]
                        (put! c response)
                        (close! c)))
    c))

(defn create [{:keys [machine-names]}]
  (specify {:instance      (AWS.StepFunctions.)
            :machine-names machine-names}
    Actionable
    (-perform [{:keys [instance machine-names] :as this} [_ payload]]
      (go
        (let [query (to-payload payload machine-names)]
          (<! (-execute this query)))))))
