(ns hx-frame-sockets.events
  (:require
   [hx-frame.core :as hx-frame]

   [hx-frame-sockets.utils :as u]))

(def ^{:private true} conn (atom nil))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; WS LISTENERS
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn ws-on-open
  "WebSocket onopen hander"
  [on-open-cb event]
  (hx-frame/dispatch [:socket/update-connection-status :connected])
  (on-open-cb))

(defn ws-on-close
  "WebSocket onclose hander"
  [on-close-cb event]
  (reset! conn nil)
  (hx-frame/dispatch [:socket/update-connection-status :disconnected])
  (on-close-cb))

(defn ws-on-message
  "WebSocket onmessage hander"
  [router event]
  (let [{:keys [action client-id] :as message} (u/event->message event)
        hx-frame-action (get router action)
        is-internal-error (= (:message message) "Internal server error")
        is-error (some-> action namespace (= "error"))]

    (if (true? is-internal-error)
      (js/console.error message)
      (do
        (u/validate-message! message "Invalid inbound message")

        (when (and (some? client-id)
                   (false? is-error))
          (hx-frame/dispatch [:socket/fulfill-request-success message]))

        (when (and (some? client-id)
                   (true? is-error))
          (hx-frame/dispatch [:socket/fulfill-request-failure message]))

        (when (some? hx-frame-action)
          (hx-frame/dispatch [hx-frame-action message]))

        (when (and (nil? hx-frame-action)
                   (false? is-error))
          (throw (js/Error.
                  (str "Inbound message '" action
                       "' has no matching handler"))))))))

(defn ws-on-error
  "WebSocket onerror hander"
  [event]
  (js/console.error "On Error" event))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; fulfill-request
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn fulfill-request-success
  [{:keys [db]} [_ {:keys [client-id] :as a}]]
  (let [event (get-in db [:hx-frame-sockets :messages client-id])]
    (cond-> {:db db}
      (some? event)
      (update-in [:db :hx-frame-sockets :messages] dissoc client-id))))

(defn fulfill-request-failure
  [{:keys [db]} [_ {:keys [client-id action]}]]
  (let [event (get-in db [:socket :messages client-id])]
    (cond-> {:db db}
      (some? event)
      (update-in [:db :hx-frame-sockets :messages] dissoc client-id)

      (qualified-keyword? event)
      (assoc :dispatch [(-> event
                            u/ns-kw->ns-str
                            (str "-failure")
                            (keyword))
                        action]))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; send-message
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn send-message
  "Send a message on the socket"
  [{:keys [db]} [_ {:keys [track] :as message}]]
  (let [tracking-id (str (random-uuid))]
    {:db (cond-> db
           (some? track)
           (assoc-in [:hx-frame-sockets :messages tracking-id] track))
     :ws-message (cond-> message
                   (some? track)
                   (assoc :track tracking-id))}))

(defn send-ws-message
  "Proxies message to :socket/send-message"
  [payload]
  (hx-frame/dispatch [:socket/send-message payload]))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; update-connection-state
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn update-connection-status
  [db [_ status]]
  (assoc-in db [:hx-frame-sockets :status] status))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; EFFECTS
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn ws-message
  [{:keys [track before-send after-send mock] :as message}]
  (let [formatted-message (cond-> (select-keys message [:action :data])
                            (some? track)
                            (assoc :client-id track))]

    (u/validate-message! formatted-message "Invalid outbound message")

    ;; TODO This technically does not guarantee that before and after fire in
    ;; that order - they should be wrapped if this feature is required.
    (when (some? before-send)
      (hx-frame/dispatch (conj before-send formatted-message)))

    (if (some? mock)
      (let [{:keys [time message router]
             :or {time 50}} mock]
        (js-invoke js/window "setTimeout"
                   #(ws-on-message router
                                   ;; Simulate a WS event
                                   #js {:data (u/clj->JSON
                                               (cond-> message
                                                 (some? track)
                                                 (assoc :client-id track)))})
                   time))

      (if (some? @conn)
        (->> (update formatted-message :action u/ns-kw->ns-str)
             u/clj->JSON
             (js-invoke @conn "send"))
        (js/console.error "hx-frame-socket: No socket connection found.")))

    (when (some? after-send)
      (hx-frame/dispatch (conj after-send formatted-message)))))

(defn init
  [{:keys [db]} [_ config]]
  {:db db :socket/init config})

(defn init-effect
  [{:keys [on-open on-close router ws-url credentials]
    :or {on-open identity
         on-close identity}}]
  (let [ws (js/WebSocket. (cond-> ws-url
                            (some? credentials)
                            (str "?token=" credentials)))]
    (doto ws
      (aset "onopen" (partial ws-on-open on-open))
      (aset "onclose" (partial ws-on-close on-close))
      (aset "onmessage" (partial ws-on-message router))
      (aset "onerror" ws-on-error))

    (reset! conn ws)))

(defn close
  [{:keys [db]} _]
  {:db db
   :socket/close nil})

(defn close-effect
  [_]
  (when (instance? js/WebSocket @conn)
    (js-invoke @conn "close")))

(defn register-events
  []
  (hx-frame/register-event-fx :socket/init init)
  (hx-frame/register-effect :socket/init init-effect)
  (hx-frame/register-event-fx :socket/close close)
  (hx-frame/register-effect :socket/close close-effect)

  (hx-frame/register-effect :ws-message ws-message)
  (hx-frame/register-effect :send-ws-message send-ws-message)

  (hx-frame/register-event-db :socket/update-connection-status
                              update-connection-status)
  (hx-frame/register-event-fx :socket/fulfill-request-success
                              fulfill-request-success)
  (hx-frame/register-event-fx :socket/fulfill-request-failure
                              fulfill-request-failure)
  (hx-frame/register-event-fx :socket/send-message
                              send-message))
