(ns obis-shared.entity.service-requests
  (:import [java.util Date])
  (:use obis-shared.utils.gen-class)
  (:use [clojure.data.json :only (json-str write-json read-json)])
  (:require [clj-http.client :as client]
            [obis-shared.entity.http-json :as http]
            [obis-shared.entity.config :as config]
            [clojure.walk :as walk]
            [obis-shared.entity.utils :as u]))

(def from-project-to-service-request-relationship "project_service_request_membership")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; CRUD FUNCTIONS FOR SERVICE-REQUESTS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;TODO: CREATE
;;TODO: DELETE
 
(defn delete
  [id]
  (http/delete ["service_requests" id]))

;; READ
(defn all
  []
  (http/get ["service_requests"]))

(defn get-service-request
  [id]
  (http/get ["service_requests" id]))

(defn get-by-friendly-id
  [friendly-id]
  (http/get ["identifiers" "service_request_id" friendly-id]))

(defn relationships
  [id]
  (http/get ["service_requests" id "relationships"]))

(defn relationships-by-type
  [sr-id relationship-type]
  (let [r (relationships sr-id)]
    (doall (filter #(= (:relationship_type %) relationship-type) r))))

;;TODO: MEMOIZE
(defn related-project
  [sr-id]
  (:to (first (relationships-by-type sr-id from-project-to-service-request-relationship))))

;;TODO: POJECTERISH
;;TODO: Look at whether the Couch View can be optimized even more.
(defn ssrs-for-organizations
  [organization-ids]
  (-> (client/post (str (config/couchdb-url) "_design/org_ssr/_view/org_ssr")
                   {:body (json-str {:keys organization-ids})
                    :content-type :json
                    "Content-Type:" "application/json"
                    :accept :json})
      (:body)
      (read-json)
      (:rows)))

;;UPDATE
(defn update-service-request
  ([obis-id attrs]  
     (update-service-request obis-id attrs {}))
  ([obisid attrs identifiers]
     (http/put ["service_requests" obisid]
            {:attributes attrs,
             :identifiers identifiers})))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; GETTERS FOR SERVICE-REQUEST
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn obis-id
  [sr]
  (get-in sr [:identifiers :OBISID]))

(defn service-request-status
  [sr]
  (get-in sr [:attributes :status]))

(defn get-sub-service-request
  [ssr-id sr]
  (let [ssrs (get-in sr [:attributes :sub_service_requests])]
    (get ssrs (keyword ssr-id))))

(defn get-line-items
  [sr]
  (get-in sr [:attributes :line_items]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; GETTERS FOR SUB-SERVICE-REQUEST
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn sub-service-request-status-and-status-date
  [ssr-id sr]
  (let [ssr (get-sub-service-request ssr-id sr)]
    (if (nil? (:status ssr))
      nil
      [(:status ssr)
       (:status_date ssr)])))

(defn service-ids-for-sub-service-request
  [ssr-id sr]
  (let [line-items (get-line-items sr)]
    (doall (map :service_id
                (doall (filter #(= (:sub_service_request_id %) ssr-id)
                        line-items)) ))))

(defn sub-service-request-past-status-list
  [ssr-id sr]
  (let [ssr (get-sub-service-request ssr-id sr)]
   (get ssr :past_statuses [])))

;;TODO: Change this to use sub-service-request-status-and-status-date
(defn sub-service-request-statuses
  [sr]
  (let [ssrs (get-in sr [:attributes :sub_service_requests])]
    (into {} (for [ssr-id (keys ssrs)]
               [ssr-id
                (get-in ssrs [ssr-id :status])]))))

(defn get-line-items-for-sub-service-request
  [sr sub-service-request-id]
  (let [li (get-line-items sr)
        li-grouped (group-by :sub_service_request_id li)]
    (get li-grouped sub-service-request-id)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; GENERIC MODIFY
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn modify-service-request
  [sr key value]
  (assoc-in sr [:attributes
                (keyword key)]
            value))

(defn modify-sub-service-request
  [ssr-id sr key value]
  (assoc-in sr [:attributes
                :sub_service_requests
                (keyword ssr-id)
                (keyword key)]
            value))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; SUB SERVICE REQUEST STATUS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn modify-sub-service-request-past-status-list
  [ssr-id sr]
  (let [status-and-date (sub-service-request-status-and-status-date ssr-id sr)
        past-status-list (sub-service-request-past-status-list ssr-id sr)]
    (if (nil? status-and-date)
      sr
      (modify-sub-service-request ssr-id
                                  sr
                                  "past_statuses"
                                  (conj past-status-list status-and-date)))))

;;TODO: POJECTIFIED
(defn modify-sub-service-request-status
  [ssr-id sr new-status]
  (modify-sub-service-request ssr-id
                              sr
                              "status"
                              new-status))

(defn modify-service-request-status
  [sr new-status]
  (modify-service-request sr "status" new-status))

;;BUSINESS RULES
;; If all sub-service-requests are DRAFT or SUBMITTED, then the options for a
;; USER are VIEW and EDIT.

;; If any sub-service-request is in a status other than DRAFT or
;; SUBMITTED, then the options are VIEW only.
(defn nil-or-submitted-or-draft
  [status]
  (or (= "SUBMITTED" status)
      (= nil status)
      (= "DRAFT" status)))

(defn distinct-ssr-statuses
  [sr]
  (set (vals (sub-service-request-statuses sr))))

(defn can-service-request-be-edited
  [sr]
  (every? nil-or-submitted-or-draft (distinct-ssr-statuses sr)))

(defn all-ssr-statuses-complete
  [sr]
  (every? #{"COMPLETE"} (distinct-ssr-statuses sr)))
