(ns obis-shared.entity.api
  (:use obis-shared.utils.gen-class)
  (:use [clojure.data.json :only (json-str write-json read-json)])
  (:require [obis-shared.entity :as e]
            [obis-shared.entity.identity :as i]
            [obis-shared.entity.services :as s]
            [obis-shared.entity.organizations :as o]
            [obis-shared.entity.service-requests :as sr]
            [clojure.walk :as walk]
            [obis-shared.entity.utils :as u]))

(def ^{:dynamic true} *sr* {})
(def ^{:dynamic true} *ssr-id* "")
(def ^{:dynamic true} *line-items* [])
(def ^{:dynamic true} *line-item* nil)
(def ^{:dynamic true} *visits* [])

(defn with-line-items-fn
  [func]
  (binding [*line-items* (sr/get-line-items *sr*)]
    (func)))

(defmacro with-line-items
  [& body]
  `(with-line-items-fn (fn [] ~@body)))

(defn with-sub-service-request-fn
  [ssr-id sr-id func]
  (binding [*sr* (sr/get-service-request sr-id)
            *ssr-id* ssr-id]
    (func)))

(defmacro with-sub-service-request
  [ssr-id sr-id & body]
  `(with-sub-service-request-fn ~ssr-id ~sr-id (fn [] ~@body)))

;; TODO: 
;; (defmacro update-service-request
;;   []
;;   `(let [new-sr# ~@body]
;;      (sr/update-service-request sr-id
;;                                 (:attributes new-sr)
;;                                 (:identifiers new-sr))))

(defn match-service-id-in-line-items
  [service-id line-items]
  (first
   (doall
    (keep-indexed #(if (= (:service_id %2) service-id) %1) line-items))))

(defn replace-line-item-in-line-items
  [line-item-index new-line-item line-items]
  (assoc line-items line-item-index new-line-item))

(defn replace-line-items-in-service-request
  [new-line-items sr]
  (assoc-in sr [:attributes :line_items] new-line-items))

(defn update-service-request-with-replacement-line-item
  [replacement-line-item sr-id]
  (let [sr (sr/get-service-request sr-id)
        line-items (sr/get-line-items sr)
        line-item-index (match-service-id-in-line-items (:service_id replacement-line-item)
                                                        line-items)
        new-line-items (replace-line-item-in-line-items line-item-index
                                                        replacement-line-item
                                                        line-items)
        new-sr (replace-line-items-in-service-request new-line-items sr)]
    (sr/update-service-request sr-id
                               (:attributes new-sr)
                               (:identifiers new-sr))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; SERVICE CHANGES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; GENERIC LINE ITEM METHODS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn index-of-line-item-for-service-id
  [service-id]
  (match-service-id-in-line-items service-id *line-items*))

(defn remove-line-item-from-line-items
  [service-id line-items]
  (doall
   (remove #(= (:service_id %) service-id)  line-items)))

;; Delete
;; - sub_service_request_id (string - friendly id)
;; - service_id (string - long ass obis-id)
(defn delete-line-item
  "Removes a line-item."
  [sr-id service-id]
  (let [sr (sr/get-service-request sr-id)
        line-items (sr/get-line-items sr)
        new-line-items (remove-line-item-from-line-items service-id
                                                         line-items)
        new-sr (replace-line-items-in-service-request new-line-items
                                                      sr)]
    (sr/update-service-request sr-id
                               (:attributes new-sr)
                               (:identifiers new-sr))))

;;TODO in-process-date update
;;TODO complete-date update

(defn update-line-item-quantity-or-subject-count
  [index new-line-items quantity]
  (let [line-item (get new-line-items index)]
    (if (contains? line-item :quantity)
      (assoc-in new-line-items [index :quantity] quantity)
      (assoc-in new-line-items [index :subject_count] quantity))))

(defn update-line-item-service-and-quantity-in-service-request
  ""
  [old-service-id new-service-id quantity]
  (let [i (index-of-line-item-for-service-id old-service-id)
        new-line-items *line-items*
        new-line-items (assoc-in new-line-items [i :service_id] new-service-id)
        new-line-items (update-line-item-quantity-or-subject-count i new-line-items quantity)
        new-sr (replace-line-items-in-service-request new-line-items
                                                      *sr*)]
    new-sr))

(defn change-line-item-service-and-quantity
  "Updates a line-item service and quantity."
  [sr-id ssr-id new-service-id old-service-id quantity]
  (with-sub-service-request ssr-id sr-id
    (with-line-items
      (let [new-sr (update-line-item-service-and-quantity-in-service-request old-service-id
                                                                             new-service-id
                                                                             quantity)]
        (sr/update-service-request sr-id
                                   (:attributes new-sr)
                                   (:identifiers new-sr))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; PER-PATIENT PER-VISIT
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn build-new-per-patient-per-visit-line-item
  [ssr-id optional quantity service-id visits subject-count is-one-time-fee]
  {:sub_service_request_id ssr-id
   :optional optional
   :quantity quantity
   :service_id service-id
   :visits visits
   :subject_count subject-count
   :is_one_time_fee is-one-time-fee})

(defn create-visit
  [quantity billing]
  {:quantity quantity :billing billing})

(defn create-new-per-patient-per-visit-line-item
  ""
  [service-id]
  (let [subject-count (get-in *sr* [:attributes :subject_count])
        visit-count (get-in *sr* [:attributes :visit_count])
        visits (take visit-count (repeat (create-visit 0 "")))
        new-line-items (conj *line-items* {:sub_service_request_id *ssr-id*
                                           :optional true
                                           :subject_count subject-count
                                           :service_id service-id
                                           :visits visits
                                           :quantity 0})]
    (replace-line-items-in-service-request new-line-items
                                           *sr*)))

(defn add-per-patient-per-visit-line-item
  "Add a new per-patient/per-visit line item."
  [ssr-id sr-id service-id]
  (with-sub-service-request ssr-id sr-id
    (with-line-items
      (let [new-sr (create-new-per-patient-per-visit-line-item service-id)]
        (sr/update-service-request sr-id
                                   (:attributes new-sr)
                                   (:identifiers new-sr))))))

(defn ordinality-of-per-patient-per-visit-line-items
  [service-request]
  (doall
    (keep-indexed #(if (contains? %2 :visits) %1) (sr/get-line-items service-request))))

(defn add-visit-to-appropriate-line-items
  [service-request visit-number]
  (let [per-patient-per-visit-line-items (ordinality-of-per-patient-per-visit-line-items @service-request)
        blank-visit (create-visit 0 "")]
    (doall
     (map
      #(swap! service-request update-in [:attributes :line_items % :visits] u/insert-into-vector visit-number blank-visit)
      per-patient-per-visit-line-items))
    service-request))

(defn increase-visit-count
  [service-request]
  (swap! service-request update-in  [:attributes :visit_count] inc))

(defn create-per-patient-per-visit-visit
  [visit-number]
  (let [service-request (atom *sr*)]
    (increase-visit-count service-request)
    (add-visit-to-appropriate-line-items service-request visit-number)
    service-request))

(defn add-per-patient-per-visit-visit
  "Add a new per-patient/per-visit visit."
  [ssr-id sr-id visit-number]
  (with-sub-service-request ssr-id sr-id
    (let [new-sr (create-per-patient-per-visit-visit visit-number)]
        (sr/update-service-request sr-id
                                   (:attributes new-sr)
                                   (:identifiers new-sr)))))

(defn remove-visit-from-appropriate-line-items
  [service-request visit-number]
  (let [per-patient-per-visit-line-items (ordinality-of-per-patient-per-visit-line-items @service-request)
        blank-visit (create-visit 0 "")]
    (doall
     (map
      #(swap! service-request update-in [:attributes :line_items % :visits] u/remove-from-vector visit-number)
      per-patient-per-visit-line-items))
    service-request))

(defn decrease-visit-count
  [service-request]
  (swap! service-request update-in  [:attributes :visit_count] dec))

(defn delete-per-patient-per-visit-visit
  [visit-number]
  (let [service-request (atom *sr*)]
    (decrease-visit-count service-request)
    (remove-visit-from-appropriate-line-items service-request visit-number)
    service-request))

(defn remove-per-patient-per-visit-visit
  "Remove a new per-patient/per-visit visit."
  [ssr-id sr-id visit-number]
  (with-sub-service-request ssr-id sr-id
    (let [new-sr (delete-per-patient-per-visit-visit visit-number)]
        (sr/update-service-request sr-id
                                   (:attributes new-sr)
                                   (:identifiers new-sr)))))

;; TODO: FIX THIS TO USE THE NEW
;; per-patient/per-visit services
;; - is_one_time_fee (boolean)
;; - subject_count (integer)
;; - optional (boolean)
;; - sub_service_request_id (string - friendly id)
;; - service_id (string - long ass obis-id)
;; - quantity (integer)
;; - visits (array)-
;;   * quantity (integer)
;;   * billing (string "R"/"N")
(defn update-per-patient-per-visit-line-item
  [sr-id ssr-id service-id subject-count
   optional quantity visits is-one-time-fee]
  (-> (build-new-per-patient-per-visit-line-item ssr-id
                                                 optional
                                                 quantity
                                                 service-id
                                                 visits
                                                 subject-count
                                                 is-one-time-fee)
     (update-service-request-with-replacement-line-item sr-id)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ONE TIME FEE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn build-one-time-fee-line-item
  ""
  [ssr-id optional quantity service-id fulfillment is-one-time-fee]
  {:sub_service_request_id ssr-id
   :optional optional
   :quantity quantity
   :service_id service-id
   :fulfillment fulfillment
   :is_one_time_fee is-one-time-fee})

(defn add-one-time-fee-line-item
  ""
  ([ssr-id sr-id service-id in-process-date complete-date]
     (add-one-time-fee-line-item ssr-id sr-id service-id 1 in-process-date complete-date))
  ([ssr-id sr-id service-id quantity in-process-date complete-date]
     (with-sub-service-request ssr-id sr-id
       (with-line-items
         (let [new-line-items (conj *line-items* {:sub_service_request_id ssr-id
                                                  :optional true
                                                  :quantity quantity
                                                  :service_id service-id
                                                  :in_process_date in-process-date
                                                  :complete_date complete-date})
               new-sr (replace-line-items-in-service-request new-line-items
                                                             *sr*)]
           (sr/update-service-request sr-id
                                      (:attributes new-sr)
                                      (:identifiers new-sr)))))))

(defn change-one-time-fee-line-item
  ""
  [old-service-id new-service-id quantity fulfillment in-process complete]
  (let [i (index-of-line-item-for-service-id old-service-id)
        new-line-items *line-items*
        new-line-items (assoc-in new-line-items [i :service_id] new-service-id)
        new-line-items (update-line-item-quantity-or-subject-count i new-line-items quantity)
        new-line-items (assoc-in new-line-items [i :fulfillment] fulfillment)
        new-line-items (assoc-in new-line-items [i :in_process_date] in-process)
        new-line-items (assoc-in new-line-items [i :complete_date] complete)
        new-sr (replace-line-items-in-service-request new-line-items
                                                      *sr*)]
    new-sr))

;; Non per-patient/per-visit services (One time fees)
;; - is_one_time_fee (boolean)
;; - optional (boolean)
;; - sub_service_request_id (string - friendly id)
;; - service_id (string - long ass obis-id)
;; - quantity (integer)
(defn update-one-time-fee-line-item
  "Update a one-time-fees service and quantity."
  [sr-id ssr-id new-service-id old-service-id quantity fulfillment in-process complete]
  (with-sub-service-request ssr-id sr-id
    (with-line-items
      (let [new-sr (change-one-time-fee-line-item old-service-id
                                                  new-service-id
                                                  quantity
                                                  fulfillment
                                                  in-process
                                                  complete)]
        (sr/update-service-request sr-id
                                   (:attributes new-sr)
                                   (:identifiers new-sr))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; SUBSIDY
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn change-subsidy-amount
  [subsidy-id subsidy-amount-in-cents]
  (assoc-in *sr*
            [:attributes :subsidies (keyword subsidy-id) :pi_contribution]
            subsidy-amount-in-cents))

(defn update-subsidy-amount
  [ssr-id sr-id subsidy-id subsidy-amount-in-cents]
  (with-sub-service-request ssr-id sr-id
    (let [new-sr (change-subsidy-amount subsidy-id subsidy-amount-in-cents)]
      (sr/update-service-request sr-id
                                 (:attributes new-sr)
                                 (:identifiers new-sr)))))

(gen-class-with-exports)

