(ns kubernetes-api.internals.client
  (:require [camel-snake-kebab.core :as csk]
            [clojure.string :as string]
            [kubernetes-api.misc :as misc]
            [kubernetes-api.internals.meta :as meta]))

(defn pascal-case-routes [k8s]
  (update k8s :handlers
          (fn [handlers]
            (mapv #(update % :route-name csk/->PascalCase) handlers))))

(defn patch-http-verb [k8s]
  (assoc k8s :handlers
         (mapv (fn [{:keys [method] :as handler}]
                 (if (#{"apply" "patch"} (namespace method))
                   (assoc handler :method :patch)
                   handler))
               (:handlers k8s))))

(defn transform [k8s]
  (-> k8s
      pascal-case-routes
      patch-http-verb))

(defn swagger-definition-for-route [k8s route-name]
  (->> (:handlers k8s)
       (misc/find-first #(= route-name (:route-name %)))
       :swagger-definition))

(defn handler-kind [handler]
  (-> handler :swagger-definition :x-kubernetes-group-version-kind :kind keyword))

(defn handler-group [handler]
  (-> handler :swagger-definition :x-kubernetes-group-version-kind :group))

(defn handler-version [handler]
  (-> handler :swagger-definition :x-kubernetes-group-version-kind :version))

(defn handler-action [handler]
  (-> handler :swagger-definition :x-kubernetes-action keyword))

(defn all-namespaces-route? [route-name]
  (string/ends-with? (name route-name) "ForAllNamespaces"))

(defn scale-resource [route-name]
  (second (re-matches #".*Namespaced([A-Za-z]*)Scale" (name route-name))))

(defn status-route? [route-name]
  (re-matches #".*Status(JsonPatch|StrategicMerge|JsonMerge|ApplyServerSide)?" (name route-name)))

(defn kind
  "Returns a kubernetes-api kind. Similar to handler-kind, but deals with some
  corner-cases. Returns a keyword, that is namespaced only if there's a
  subresource.

  Example:
  Deployment/Status"
  [{:keys [route-name] :as handler}]
  (let [kind (some-> (handler-kind handler) name)]
    (cond
      (status-route? route-name) (keyword kind "Status")
      (string/ends-with? (name route-name) "Scale") (keyword (scale-resource route-name) "Scale")
      :else (keyword kind))))

(defn action
  "Return a kubernetes-api action. Similar to handler-action, but tries to be
  unique for each kind.

  Example:
  :list-all"
  [{:keys [route-name method] :as handler}]
  (cond
    (re-matches #"Create.*NamespacedPodBinding" (name route-name)) :pod/create
    (re-matches #"Connect.*ProxyWithPath" (name route-name)) (keyword "connect.with-path" (name method))
    (re-matches #"Connect.*Proxy" (name route-name)) (keyword "connect" (name method))
    (re-matches #"Connect.*" (name route-name)) (keyword "connect" (name method))
    (re-matches #"ReplaceCertificates.*CertificateSigningRequestApproval" (name route-name)) :replace-approval
    (re-matches #"Read.*NamespacedPodLog" (name route-name)) :misc/logs
    (re-matches #"Replace.*NamespaceFinalize" (name route-name)) :misc/finalize
    (all-namespaces-route? route-name) (keyword (str (name (handler-action handler)) "-all"))
    :else (handler-action handler)))

(defn find-route [k8s {:keys [all-namespaces?] :as _search-params
                       search-kind :kind
                       search-action :action}]
  (->> (:handlers k8s)
       (filter (fn [handler]
                 (and (or (= (keyword search-kind) (kind handler)) (nil? search-kind))
                      (or (= (keyword search-action) (action handler)) (nil? search-action))
                      (= (boolean all-namespaces?) (all-namespaces-route? (:route-name handler))))))
       (map :route-name)))

(defn find-handler [k8s {:keys [all-namespaces?] :as _search-params
                       search-kind :kind
                       search-action :action}]
  (->> (:handlers k8s)
       (filter (fn [handler]
                 (and (or (= (keyword search-kind) (kind handler)) (nil? search-kind))
                      (or (= (keyword search-action) (action handler)) (nil? search-action))
                      (= (boolean all-namespaces?) (all-namespaces-route? (:route-name handler))))))))

(defn version-of [k8s route-name]
  (->> (swagger-definition-for-route k8s route-name)
       :x-kubernetes-group-version-kind
       :version))

(defn group-of [k8s route-name]
  (->> (swagger-definition-for-route k8s route-name)
       :x-kubernetes-group-version-kind
       :group))

(defn kind-of [k8s route-name]
  (->> (swagger-definition-for-route k8s route-name)
       :x-kubernetes-group-version-kind
       :kind
       keyword))

(defn core-versions [k8s]
  (mapv
   #(hash-map :name ""
              :versions [{:groupVersion % :version %}]
              :preferredVersion {:groupVersion % :version %})
   (:versions (:kubernetes-api.core/core-api-versions k8s))))

(defn all-versions [k8s]
  (concat (:groups (:kubernetes-api.core/api-group-list k8s))
          (core-versions k8s)))

(defn ^:private choose-preffered-version [k8s route-names]
  (misc/find-first
   (fn [route]
     (some #(and (= (:name %) (group-of k8s route))
                 (= (:version (:preferredVersion %)) (version-of k8s route)))
           (all-versions k8s)))
   route-names))

(defn find-preferred-route [k8s search-params]
  (->> (find-route k8s search-params)
       (filter (fn [x] (not (string/ends-with? (name x) "Status"))))
       ((partial choose-preffered-version k8s))))


(defn preffered-version? [k8s handler]
  (let [preffered-route (find-preferred-route k8s {:kind   (handler-kind handler)
                                                   :action (handler-action handler)})]
    (and (= (handler-version handler) (version-of k8s preffered-route))
         (= (handler-group handler) (group-of k8s preffered-route)))))

(defn k8s-version-pattern [s]
  (when (seq s)
    (when-let [groups (re-matches #"^v(\d+)(alpha|beta|)(\d*)$" s)]
      {:raw s :ga (Integer/parseInt (nth groups 1)) :channel (nth groups 2) :postversion (when (seq (nth groups 3)) (Integer/parseInt (nth groups 3)))})))

(defn compare-channel [a b] 
  (let [channels {"alpha" 2 "beta" 1 "" 0}] 
    (compare (get channels a 3) (get channels b 3))))

(defn compare-k8s-versions [a b]
  (let [k8s-a (k8s-version-pattern a)
        k8s-b (k8s-version-pattern b)]
    (cond 
      (and (nil? k8s-a) (not (nil? k8s-b)))
      1

      (and (not (nil? k8s-a)) (nil? k8s-b))
      -1

      (and (nil? k8s-a) (nil? k8s-b))
      (compare a b)

      (not= (:channel k8s-a) (:channel k8s-b))
      (compare-channel (:channel k8s-a) (:channel k8s-b))


      (not= (:ga k8s-a) (:ga k8s-b))
      (compare (:ga k8s-b) (:ga k8s-a))


      :else (compare (:postversion k8s-a) (:postversion k8s-b)))))

(defn compare-group [k8s a b]
  (letfn [(preffered-version-of [group]
                                (->> (all-versions k8s)
                                     (some #(= (:name %) group))
                                     :prefferedVersion
                                     :version))]
    (compare-k8s-versions (preffered-version-of a) (preffered-version-of b))))

(defn compare-handler-version [k8s a b]
  (cond
    (not= (handler-group a) (handler-group b))
    (compare-group k8s (handler-group a) (handler-group b))

    (and (preffered-version? k8s a) (preffered-version? k8s b))
    (compare-k8s-versions (handler-version a) (handler-version b))

    (preffered-version? k8s a)
    -1

    (preffered-version? k8s b)
    1

    :else
    (compare-k8s-versions (handler-version a) (handler-version b))))

(defn kind-handlers [k8s kind]
  (->> (:handlers k8s)
       (filter (fn [handler] (= kind (kind handler))))
       (sort compare-handler-version)))

(defn kind-handler [k8s kind] 
  (first (kind-handlers k8s kind)))


(defn from-api-version? [api handler]
  (let [{:keys [group version]} (meta/group-version api)] 
    (and (= group (handler-group handler))
         (= version (handler-version handler)))))

(defn find-explicit-api-route [k8s api search-params]
  (->> (find-route k8s search-params)
       (filter (partial from-api-version? api))
       first))

(defn find-version-priority-handler [k8s search-params] 
  (->> (find-handler k8s search-params)
       (filter (fn [x] (not (string/ends-with? (name (:route-name x)) "Status"))))
       (sort (partial compare-handler-version k8s))
       first))

(defn find-version-priority-route [k8s search-params]
  (:route-name (find-version-priority-handler k8s search-params)))

(defn select-route [k8s search-params]
  (if-let [api (or (:api search-params)
                   (get-in search-params [:request :apiVersion]))]
    (find-explicit-api-route k8s api search-params)
    (find-version-priority-route k8s search-params)))
