(ns pos5.user
  (:require
   [pos5.request :as req]
   [pos5.url :as url]
   #?@(:clj  [[clojure.spec.alpha :as s]]
       :cljs [[cljs.spec.alpha :as s]])))

;; -------------------------------------------------------------------
;; user client spec
;; -------------------------------------------------------------------

(s/def ::host-url
  string?)

(s/def ::content-type
  #{:transit :json})

;; -------------------------------------------------------------------
;; protocols
;; -------------------------------------------------------------------

(defprotocol IUserClient
  (-register [this user])
  (-erase [this username password])
  (-authenticate [this username password])
  (-reset-password [this username password new-password]))

;; -------------------------------------------------------------------
;; user client
;; -------------------------------------------------------------------

(defrecord UserClient [host-url content-type]
  IUserClient
  (-register [_ user]
    (req/POST (url/create-user-url host-url)
              (-> {}
                  (req/wrap-content-type content-type)
                  (assoc :params user))))
  (-erase [_ username password]
    (req/DELETE (url/target-user-url host-url username)
                (-> {}
                    (req/wrap-content-type content-type)
                    (req/wrap-http-basic-auth username password))))
  (-authenticate [_ username password]
    (req/POST (url/target-user-url host-url username)
              (-> {}
                  (req/wrap-content-type content-type)
                  (req/wrap-http-basic-auth username password))))
  (-reset-password [_ username password new-password]
    (req/PUT (url/target-user-url host-url username)
             (-> {}
                 (req/wrap-content-type content-type)
                 (req/wrap-http-basic-auth username password)
                 (assoc :params {:password new-password})))))

(defn make-user-client
  ([host-url content-type]
   (->UserClient (s/assert ::host-url host-url)
                 (s/assert ::content-type content-type)))
  ([host-url]
   (make-user-client host-url :json)))

(defn register
  [iuser-client user]
  (-register iuser-client user))

(defn erase
  [iuser-client username password]
  (-erase iuser-client username password))

(defn authenticate
  [iuser-client username password]
  (-authenticate iuser-client username password))

(defn reset-password
  [iuser-client username password new-password]
  (-reset-password iuser-client username password new-password))
